const { sendErrorResponse, sendResponse, } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// --> Create Day Model
module.exports.createDayModel = async (req, res) => {
    try {
    const {
      day_model_id_external,
      external_name,
      planned_hours,
      planned_hours_formated,
      country,
      shift_classification,
      time_recording_variant,
      cross_midnight_allowed = 2,
      non_working_day = 2
    } = req.body;

    if (
      !day_model_id_external ||
      !external_name ||
      !planned_hours ||
      !planned_hours_formated ||
      !country ||
      !time_recording_variant || 
      !cross_midnight_allowed ||
      !non_working_day
    ) {
      const errorMessage = "Missing required fields!";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Duplicate check
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.day_model}
       WHERE day_model_id_external = ? AND is_active = 1`,
      [day_model_id_external]
    );
    if (existingRecord.length > 0) {
      return sendErrorResponse(
        res,
        "Day Model already exists",
        "Day Model already exists"
      );
    }

    // Picklist check
    const checkVariant = await performQuery(
      `SELECT * FROM ${tables.picklist_master}
       WHERE picklist_id = 19 AND id = ? AND is_deleted = 2`,
      [time_recording_variant]
    );
    if (checkVariant.length === 0) {
      return sendErrorResponse(res, "Invalid time recording variant", "Invalid time recording variant");
    }

    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    await performQuery(
      `INSERT INTO ${tables.day_model} SET ?`,
      {
        sequence_no: 1,
        day_model_id_external,
        external_name,
        planned_hours,
        planned_hours_formated,
        country,
        shift_classification,
        time_recording_variant,
        cross_midnight_allowed,
        non_working_day,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Day Model created successfully", 200);

  } catch (error) {
    console.error("Error While Creating Day Model", error);
    return sendErrorResponse(res, error, "Error while creating Day Model");
  }
};


// --> Get Day Model
module.exports.getDayModel = async (req, res) => {
    try {
    const { id, day_model_id_external } = req.query;

    let query = `
      SELECT
        dm.id,
        dm.sequence_no,
        dm.day_model_id_external,
        dm.external_name,
        dm.planned_hours,
        dm.planned_hours_formated,
        dm.country,
        dm.shift_classification,
        dm.time_recording_variant,
        pm.picklist_option AS time_recording_variant_value,
        dm.cross_midnight_allowed,
        dm.non_working_day
      FROM ${tables.day_model} dm
      LEFT JOIN ${tables.picklist_master} pm
        ON dm.time_recording_variant = pm.id
        AND pm.picklist_id = 19
        AND pm.is_deleted = 2
      WHERE dm.is_active = 1
    `;

    if (id) query += ` AND dm.id = ${id}`;
    if (day_model_id_external) query += ` AND dm.day_model_id_external = '${day_model_id_external}'`;

    query += ` ORDER BY dm.id DESC`;

    const result = await performQuery(query);
    return sendResponse(res, result, "Day Model fetched successfully", 200);

  } catch (error) {
    console.error("Error While Fetching Day Model", error);
    return sendErrorResponse(res, error, "Error While Fetching Day Model");
  }
}

// --> Update Day Model 
module.exports.updateDayModel = async (req, res) => {
    try {
    const {
      id,
      external_name,
      planned_hours,
      planned_hours_formated,
      country,
      shift_classification,
      time_recording_variant,
      cross_midnight_allowed = 0,
      non_working_day = 0
    } = req.body;

    if (
      !id ||
      !external_name ||
      planned_hours === undefined ||
      !planned_hours_formated ||
      !country ||
      !time_recording_variant
    ) {
      const errorMessage = "Missing required fields!";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.day_model} WHERE id = ? AND is_active = 1`,
      [id]
    );
    if (existingRecord.length === 0) {
      return sendErrorResponse(res, "Day Model not found", "Day Model not found");
    }

    // Picklist check
    const checkVariant = await performQuery(
      `SELECT * FROM ${tables.picklist_master}
       WHERE picklist_id = 19 AND id = ? AND is_deleted = 2`,
      [time_recording_variant]
    );
    if (checkVariant.length === 0) {
      return sendErrorResponse(res, "Invalid time recording variant", "Invalid time recording variant");
    }

    // Check for duplicate on external_name in same country (excluding current record)
    const duplicateCheck = await performQuery(
      `SELECT * FROM ${tables.day_model} WHERE external_name = ? AND country = ? AND id <> ?`,
      [external_name, country, id]
    );
    if (duplicateCheck.length > 0) {
      return sendErrorResponse(res, "Duplicate external_name in the same country", "Duplicate external_name in the same country");
    }

    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Deactivate old record
    await performQuery(
      `UPDATE ${tables.day_model} SET ? WHERE id = ?`,
      [{
        is_active: 2,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }, id]
    );

    // Insert new record
    await performQuery(
      `INSERT INTO ${tables.day_model} SET ?`,
      {
        sequence_no: Number(existingRecord[0].sequence_no) + 1,
        day_model_id_external: existingRecord[0].day_model_id_external,
        external_name,
        planned_hours,
        planned_hours_formated,
        country,
        shift_classification,
        time_recording_variant,
        cross_midnight_allowed,
        non_working_day,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Day Model updated successfully", 200);

  } catch (error) {
    console.error("Error While Updating Day Model", error);
    return sendErrorResponse(res, error, "Error while updating Day Model");
  }
}