const { sendErrorResponse, sendResponse, } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// --> Create Time Profile
module.exports.createTimeProfile = async (req, res) => {
      try {
    const {
      time_profile_id_external,
      extrenal_name,
      effective_start_date,
      main_absence_time_type,
      main_attendance_time_type,
      main_break_time_type,
      country,
      time_recording_variant
    } = req.body;

    if (
      !time_profile_id_external ||
      !extrenal_name ||
      !effective_start_date ||
      !country ||
      !time_recording_variant
    ) {
      return sendErrorResponse(res, "Missing required fields!", "Missing required fields!");
    }

    // Duplicate check
    const existing = await performQuery(
      `SELECT * FROM ${tables.time_profile}
       WHERE time_profile_id_external = ? AND is_active = 1`,
      [time_profile_id_external]
    );
    if (existing.length > 0) {
      return sendErrorResponse(res, "Time Profile already exists", "Time Profile already exists");
    }

    // Picklist check
    const checkVariant = await performQuery(
      `SELECT * FROM ${tables.picklist_master}
       WHERE picklist_id = 19 AND id = ? AND is_deleted = 2`,
      [time_recording_variant]
    );
    if (checkVariant.length === 0) {
      return sendErrorResponse(res, "Invalid time recording variant", "Invalid time recording variant");
    }

    // Get current system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    await performQuery(
      `INSERT INTO ${tables.time_profile} SET ?`,
      {
        sequence_no: 1,
        time_profile_id_external,
        extrenal_name,
        effective_start_date,
        main_absence_time_type,
        main_attendance_time_type,
        main_break_time_type,
        country,
        time_recording_variant,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Time Profile created successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while creating Time Profile");
  }
};

// --> Get Time Profile
module.exports.getTimeProfile = async (req, res) => {
    try {
    const { id, time_profile_id_external } = req.query;

    let query = `
      SELECT
        tp.*,
        pm.picklist_option AS time_recording_variant_value
      FROM ${tables.time_profile} tp
      LEFT JOIN ${tables.picklist_master} pm
        ON tp.time_recording_variant = pm.id
        AND pm.picklist_id = 22
        AND pm.is_deleted = 2
      WHERE tp.is_active = 1
    `;

    if (id) query += ` AND tp.id = ${id}`;
    if (time_profile_id_external)
      query += ` AND tp.time_profile_id_external = '${time_profile_id_external}'`;

    query += ` ORDER BY tp.id DESC`;

    const result = await performQuery(query);
    return sendResponse(res, result, "Time Profile fetched successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while fetching Time Profile");
  }
};

// --> Update Time Profile
module.exports.updateTimeProfile = async (req, res) => {
     try {
    const {
      id,
      extrenal_name,
      effective_start_date,
      main_absence_time_type,
      main_attendance_time_type,
      main_break_time_type,
      country,
      time_recording_variant
    } = req.body;

    if (!id || !extrenal_name || !effective_start_date || !country || !time_recording_variant) {
      return sendErrorResponse(res, "Missing required fields!", "Missing required fields!");
    }

    const existing = await performQuery(
      `SELECT * FROM ${tables.time_profile} WHERE id = ?`,
      [id]
    );
    if (existing.length === 0) {
      return sendErrorResponse(res, "Time Profile not found", "Time Profile not found");
    }

    // Picklist check
    const checkVariant = await performQuery(
      `SELECT * FROM ${tables.picklist_master}
       WHERE picklist_id = 19 AND id = ? AND is_deleted = 2`,
      [time_recording_variant]
    );
    if (checkVariant.length === 0) {
      return sendErrorResponse(res, "Invalid time recording variant", "Invalid time recording variant");
    }

    // Get current system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    await performQuery(
      `UPDATE ${tables.time_profile} SET ? WHERE id = ?`,
      [{ is_active: 2, updated_by: req?.user?.id, updated_at: currentTime }, id]
    );

    await performQuery(
      `INSERT INTO ${tables.time_profile} SET ?`,
      {
        sequence_no: Number(existing[0].sequence_no) + 1,
        time_profile_id_external: existing[0].time_profile_id_external,
        extrenal_name,
        effective_start_date,
        main_absence_time_type,
        main_attendance_time_type,
        main_break_time_type,
        country,
        time_recording_variant,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Time Profile updated successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while updating Time Profile");
  }
};
