const { sendErrorResponse, sendResponse, } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getTimezone, getSystemTime } = require("../../functions/getTimezone.js");
const fs = require('fs');
const path = require('path');


// API to add dependant record for a person --> /personRecord/dependant POST API
module.exports.adddependantRecord = async (req, res) => {
    const uploadedFiles = [];
    try {
        // Get Variables from Request Body
        const {
            person_id,
            salutation,
            first_name,
            mid_name,
            last_name,
            relation,
            dependent_national_id,
            date_of_birth,
            effective_start_date,
            effective_end_date,
            gender,
            category,
            cost,
            insurance_company_country,
            insurance_policy_number,
            insurance_start_date,
            is_beneficiary
        } = req.body;

        // Validate Required Fields
        if (
            !person_id || !first_name || !last_name || !relation || 
            !dependent_national_id || !date_of_birth || !effective_start_date ||
            !gender || !is_beneficiary
        ) {
            return sendErrorResponse(res, null, "Missing required fields");
        }
    
        // Handle file uploads
        let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
        const profilePic = req.files?.attachment_1 ? req.files.attachment_1[0].filename : null;
        const idFrontPic = req.files?.attachment_2 ? req.files.attachment_2[0].filename : null;
        const idBackPic = req.files?.attachment_3 ? req.files.attachment_3[0].filename : null;
        if (profilePic) uploadedFiles.push(profilePic);
        if (idFrontPic) uploadedFiles.push(idFrontPic);
        if (idBackPic) uploadedFiles.push(idBackPic);

        // Return Error if idFrontPic or idBackPic is missing
        if (!idFrontPic || !idBackPic) {
            return sendErrorResponse(res, null, "Missing ID front or back picture");
        }

        // check if person exists
        const personRecord = await performQuery(
            `SELECT * FROM ${tables.per_person} WHERE id = ? AND is_deleted = 2`,
            [person_id]
        );
        if (personRecord.length === 0) {
            return sendErrorResponse(res, null, "Person not found");
        }

        // check if dependent with same national id already exists in records
        const existingDependant = await performQuery(
            `SELECT * FROM ${tables.per_dependent} WHERE dependent_national_id = ? AND is_deleted = 2`,
            [dependent_national_id]
        );
        if (existingDependant.length > 0) {
            return sendErrorResponse(res, null, "Dependant with same national ID already exists");
        }
            
        // Calculate completion and is_completed
        let total_fields = 20;
        let filled_fields = 0;

        const checkFilled = (v) => { if (v) filled_fields++; };
        [
            person_id,
            salutation,
            first_name,
            mid_name,
            last_name,
            relation,
            dependent_national_id,
            date_of_birth,
            effective_start_date,
            effective_end_date,
            gender,
            category,
            cost,
            insurance_company_country,
            insurance_policy_number,
            insurance_start_date,
            is_beneficiary,
            profilePic,
            idFrontPic,
            idBackPic
        ].forEach(checkFilled);

        const completion = Math.round((filled_fields / total_fields) * 100);
        const is_completed = completion === 100 ? 2 : 1;

        // Get System Time
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

        // Create File URL
        const profilePicURL = profilePic ? `${baseURL}/${profilePic}` : null;
        const idFrontPicURL = idFrontPic ? `${baseURL}/${idFrontPic}` : null;
        const idBackPicURL = idBackPic ? `${baseURL}/${idBackPic}` : null;

        // Insert Dependant Record
        const insertResult = await performQuery(
            `INSERT INTO ${tables.per_dependent} SET ?`,
            {
                person_id: person_id,
                salutation: salutation,
                first_name: first_name,
                mid_name: mid_name || null,
                last_name: last_name,
                display_name: `${salutation} ${first_name} ${mid_name ? mid_name + "" : ""}${last_name}`,
                relation: relation,
                dependent_national_id: dependent_national_id,
                date_of_birth: date_of_birth,
                effective_start_date: effective_start_date,
                completion: completion,
                is_completed: is_completed,
                created_by: req?.user?.id,
                created_at: currentTime,
                updated_by: req?.user?.id,
                updated_at: currentTime
            }
        );

        // Insert Dependant_Details Record
        const insertDetailsResult = await performQuery(
            `INSERT INTO ${tables.per_dependent_details} SET ?`,
            {
                related_person_id: insertResult.insertId,
                gender: gender || null,
                effective_end_date: effective_end_date || null,
                category: category || null,
                cost: cost || null,
                insurance_company_country: insurance_company_country || null,
                insurance_policy_number: insurance_policy_number || null,
                insurance_start_date: insurance_start_date || null,
                is_beneficiary: is_beneficiary || null,
                attachment_1: profilePicURL || null,
                attachment_2: idFrontPicURL || null,
                attachment_3: idBackPicURL || null
            }
        )

        // Return Success Response
        return sendResponse(res, [], "Dependant record added successfully", 201);

    } catch (error) {
        return sendErrorResponse(res, error, "Error while adding dependant record");
    }
}

// API to Update dependant record for a person --> /personRecord/dependant PUT API
module.exports.updateDependantRecord = async (req, res) => {
    uploadedFiles = [];
    try {
        // Get Variables from Request Body
        const {
            id,
            effective_end_date,
            gender,
            category,
            cost,
            insurance_company_country,
            insurance_policy_number,
            insurance_start_date,
            is_beneficiary
        } = req.body;

        // Validate Required Fields
        if (
            !id || !gender || !is_beneficiary
        ) {
            return sendErrorResponse(res, null, "Missing required fields");
        }
    
        // Handle file uploads
        let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
        const profilePic = req.files?.attachment_1 ? req.files.attachment_1[0].filename : null;
        const idFrontPic = req.files?.attachment_2 ? req.files.attachment_2[0].filename : null;
        const idBackPic = req.files?.attachment_3 ? req.files.attachment_3[0].filename : null;
        if (profilePic) uploadedFiles.push(profilePic);
        if (idFrontPic) uploadedFiles.push(idFrontPic);
        if (idBackPic) uploadedFiles.push(idBackPic);

        // get Existing Dependant Record
        const existingDependant = await performQuery(
            `
                SELECT 
                    pd.id AS dependant_id, pd.person_id, 
                    pd.salutation, pd.first_name, pd.mid_name, pd.last_name, 
                    pd.relation, pd.dependent_national_id, pd.date_of_birth, pd.effective_start_date, 
                    pdd.*
                FROM ${tables.per_dependent} pd
                left JOIN ${tables.per_dependent_details} pdd ON pd.id = pdd.related_person_id AND pdd.is_deleted = 2
                WHERE pd.id = ? AND pd.is_deleted = 2
            `,
            [id]
        );
        if (existingDependant.length === 0) {
            return sendErrorResponse(res, null, "Dependant record not found");
        }

        previousData = existingDependant[0];
            
        // Calculate completion and is_completed
        let total_fields = 20;
        let filled_fields = 0;

        const checkFilled = (v) => { if (v) filled_fields++; };
        [
            previousData.person_id,
            previousData.salutation,
            previousData.first_name,
            previousData.mid_name,
            previousData.last_name,
            previousData.relation,
            previousData.dependent_national_id,
            previousData.date_of_birth,
            previousData.effective_start_date,
            effective_end_date,
            gender,
            category,
            cost,
            insurance_company_country,
            insurance_policy_number,
            insurance_start_date,
            is_beneficiary,
            profilePic || previousData.attachment_1,
            idFrontPic || previousData.attachment_2,
            idBackPic || previousData.attachment_3
        ].forEach(checkFilled);

        const completion = Math.round((filled_fields / total_fields) * 100);
        const is_completed = completion === 100 ? 2 : 1;

        // Get System Time
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

        // Create File URL
        const profilePicURL = profilePic ? `${baseURL}/${profilePic}` : null;
        const idFrontPicURL = idFrontPic ? `${baseURL}/${idFrontPic}` : null;
        const idBackPicURL = idBackPic ? `${baseURL}/${idBackPic}` : null;

        // Mark Existing Record as Deleted to keep history
        const updateDetailsResult = await performQuery(
            `UPDATE ${tables.per_dependent_details} SET ? WHERE related_person_id = ? AND is_deleted = 2`,
            [{
                is_deleted: 1,
                updated_by: req?.user?.id,
                updated_at: currentTime,
            }, id]
        );

        // Insert New Record
        const insertDetailsResult = await performQuery(
            `INSERT INTO ${tables.per_dependent_details} SET ?`,
            {
                related_person_id: id,
                gender: gender || previousData.gender,
                effective_end_date: effective_end_date || previousData.effective_end_date,
                category: category || previousData.category,
                cost: cost || previousData.cost,
                insurance_company_country: insurance_company_country || previousData.insurance_company_country,
                insurance_policy_number: insurance_policy_number || previousData.insurance_policy_number,
                insurance_start_date: insurance_start_date || previousData.insurance_start_date,
                is_beneficiary: is_beneficiary || previousData.is_beneficiary,
                attachment_1: profilePicURL || previousData.attachment_1 || null,
                attachment_2: idFrontPicURL || previousData.attachment_2 || null,
                attachment_3: idBackPicURL || previousData.attachment_3 || null,
                created_by: req?.user?.id,
                created_at: currentTime,
                updated_by: req?.user?.id,
                updated_at: currentTime
            }
        );

        // Return Success Response
        return sendResponse(res, [], "Dependant record updated successfully", 200);

    } catch (error) {
        return sendErrorResponse(res, error, "Error while updating dependant record");
    }
}

// API to Get dependant records for a person --> /personRecord/dependant GET API
module.exports.getDependantRecords = async (req, res) => {
    try {
        // Get Query Parameters
        const { person_id, person_id_external } = req.query;

        //query
        let query = `
            Select 
                per.display_name AS person_name, per.person_id_external,
                pd.id, pd.related_person_id_external, pd.salutation, pd.first_name, pd.mid_name, pd.last_name, pd.display_name,
                pd.relation, pd.dependent_national_id, pd.date_of_birth, pd.effective_start_date, pd.completion, pd.is_completed,
                pdd.gender, pdd.effective_end_date, pdd.category, pdd.cost, pdd.insurance_company_country, pdd.insurance_policy_number,
                pdd.insurance_start_date, pdd.is_beneficiary, pdd.attachment_1, pdd.attachment_2, pdd.attachment_3
            From ${tables.per_dependent} pd
            Left Join ${tables.per_dependent_details} pdd ON pd.id = pdd.related_person_id AND pdd.is_deleted = 2
            Left Join ${tables.per_person} per ON pd.person_id = per.id
            Where pd.is_deleted = 2
        `

        // Apply Filters
        if (person_id) {
            query += ` AND pd.person_id = ${person_id} `;
        }
        else if (person_id_external) {
            query += ` AND per.person_id_external = '${person_id_external}' `;
        }

        // Sort record by id desc
        query += ` ORDER BY pd.id DESC `;

        // Fetch Dependant Records
        const dependantRecords = await performQuery(query);

        // Return Success Response
        return sendResponse(res, dependantRecords, "Dependant records fetched successfully", 200);

    } catch (error) {
        return sendErrorResponse(res, error, "Error while fetching dependant records");
    }
}