const { isUndefined } = require("lodash");
const { sendErrorResponse, sendResponse } = require("../utils");
const { getSystemTime } = require("../functions/getTimezone");
const moment = require("moment-timezone");
const { performQuery } = require("../utils/db");
const { tables } = require("../utils/tables");



// API to Create Picklist Reference --> POST /api/picklist/reference
module.exports.createPicklistReference = async (req, res) => {
    try {
        // get Variable from req body
        const {name} = req.body;

        // Validate the input
        if(isUndefined(name)){
            return sendErrorResponse(res, "Name is required", "Name is Missing", 400);
        }

        // check for duplicate name
        const duplicateCheck = await performQuery(
            `SELECT * FROM ${tables.picklist_reference} WHERE name = ? AND is_deleted = 2`,
            [name]
        );
        if(duplicateCheck.length > 0){
            return sendErrorResponse(res, "Picklist with this name already exists", "Duplicate Picklist Name", 409);
        }

        // get System Timestamp
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

        // Insert into DB
        const result = await performQuery(
            `INSERT INTO ${tables.picklist_reference} SET ?`,
            {
                name: name,
                created_at: currentTime,
                created_by: req?.user?.id,
                updated_at: currentTime,
                updated_by: req?.user?.id
            }
        );

        return sendResponse(res, {id: result.insertId}, "Picklist created successfully", 201);

    } catch (error) {
        console.log("error in creating Picklist Reference: ", error);
        return sendErrorResponse(res, error, "Failed to create picklist reference", 500);
    }
}

// API to Get Picklist Reference --> GET /api/picklist/reference
module.exports.getPicklistReference = async (req, res) => {
    try {
        // Fetch from DB
        const result = await performQuery(
            `SELECT * FROM ${tables.picklist_reference} Where is_deleted = 2 ORDER BY id DESC`
        );

        // Return Response
        return sendResponse(res, result, "Picklist fetched successfully", 200);

    } catch (error) {
        console.log("error in getting Picklist Reference: ", error);
        return sendErrorResponse(res, error, "Failed to get picklist reference", 500);
    }
}

// API to Create Picklist data --> POST /api/picklist/data
module.exports.createPicklistData = async (req, res) => {
    try {
        // get Variable from req body
        const {
            picklist_id, picklist_name, picklist_option
        } = req.body;

        // Validate the input
        if(isUndefined(picklist_id) || isUndefined(picklist_name) || isUndefined(picklist_option)){
            return sendErrorResponse(res, "All fields are required", "Missing Fields", 400);
        }

        // check if picklist exists
        const picklistCheck = await performQuery(
            `SELECT * FROM ${tables.picklist_reference} WHERE ((id = ? AND name = ?) OR name=?)  AND is_deleted = 2`,
            [picklist_id, picklist_name, picklist_name]
        );
        if(picklistCheck.length === 0){
            return sendErrorResponse(res, "Picklist not found", "Invalid Picklist ID", 404);
        }

        // Check for duplicate picklist option against the same picklist
        const duplicateCheck = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE picklist_id = ? AND picklist_option = ? AND is_deleted = 2`,
            [picklistCheck[0].id, picklist_option]
        );
        if(duplicateCheck.length > 0){
            return sendErrorResponse(res, "Picklist option already exists", "Duplicate Picklist Option", 409);
        }

        // get System Timestamp
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

        // Insert into DB
        const result = await performQuery(
            `INSERT INTO ${tables.picklist_master} SET ?`,
            {
                picklist_id: picklistCheck[0].id,
                picklist_name: picklist_name,
                picklist_option: picklist_option,
                created_at: currentTime,
                created_by: req?.user?.id,
                updated_at: currentTime,
                updated_by: req?.user?.id
            }
        );

        return sendResponse(res, {id: result.insertId}, "Picklist data created successfully", 201);

    } catch (error) {
        console.log("Error in creating Picklist Data: ", error);
        return sendErrorResponse(res, error, "Failed to create picklist data", 500);
    }
}

// API to Update Picklist data --> PUT /api/picklist/data
module.exports.updatePicklistData = async (req, res) => {
    try {
        // get Variable from req body
        const {
            id, picklist_id, picklist_name, picklist_option
        } = req.body;

        // Validate the input
        if(isUndefined(id) || isUndefined(picklist_id) || isUndefined(picklist_name) || isUndefined(picklist_option)){
            return sendErrorResponse(res, "All fields are required", "Missing Fields", 400);
        }

        // check if picklist exists
        const picklistCheck = await performQuery(
            `SELECT * FROM ${tables.picklist_reference} WHERE ((id = ? AND name = ?) OR name=?)  AND is_deleted = 2`,
            [picklist_id, picklist_name, picklist_name]
        );
        if(picklistCheck.length === 0){
            return sendErrorResponse(res, "Picklist not found", "Invalid Picklist ID", 404);
        }

        // check if picklist data exists
        const picklistDataCheck = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE id = ? AND is_deleted = 2`,
            [id]
        );
        if(picklistDataCheck.length === 0){
            return sendErrorResponse(res, "Picklist data not found", "Invalid Picklist Data ID", 404);
        }

        // Check for duplicate picklist option against the same picklist
        const duplicateCheck = await performQuery(
            `SELECT * FROM ${tables.picklist_master} WHERE picklist_id = ? AND picklist_option = ? AND is_deleted = 2 AND id <> ?`,
            [picklistCheck[0].id, picklist_option, id]
        );
        if(duplicateCheck.length > 0){
            return sendErrorResponse(res, "Picklist option already exists", "Duplicate Picklist Option", 409);
        }

        // get System Timestamp
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

        // Insert into DB
        const result = await performQuery(
            `UPDATE ${tables.picklist_master} SET ? WHERE id = ?`,
            [
                {
                    picklist_id: picklistCheck[0].id,
                    picklist_name: picklist_name,
                    picklist_option: picklist_option,
                    updated_by: req?.user?.id,
                    updated_at: currentTime
                },
                id
            ]
        );

        return sendResponse(res, {id: result.insertId}, "Picklist data Updated successfully", 200);

    } catch (error) {
        console.log("Error in Updating Picklist Data: ", error);
        return sendErrorResponse(res, error, "Failed to update picklist data", 500);
    }
}

// API to Get Picklist data --> GET /api/picklist/data
module.exports.getPicklistData = async (req, res) => {
    try {
        // get Variable from req body
        const {
            picklist_id , picklist_name
        } = req.query;

        // query
        let query = `SELECT * FROM ${tables.picklist_master} WHERE is_deleted = 2`;

        // Add filters based on query parameters
        const queryParams = [];
        if (!isUndefined(picklist_id)) {
            query += ` AND picklist_id = ${picklist_id}`;
        }
        if (!isUndefined(picklist_name)) {
            query += ` AND picklist_name = '${picklist_name}'`;
        }

        // Execute the query
        const result = await performQuery(query);
        const transformedResult = transformData(result);
        return sendResponse(res, transformedResult, "Picklist data retrieved successfully", 200);

    } catch (error) {
        console.log("Error in Getting Picklist Data: ", error);
        return sendErrorResponse(res, error, "Failed to get picklist data", 500);
    }
}


function transformData(data) {
  const map = new Map();
  data?.forEach(row => {
    // Grouping key: adjust if you want a different grouping logic
    const key = `${row.picklist_id}-${row.picklist_name}`;
    if (!map.has(key)) {
      map.set(key, {
        picklist_id: row.picklist_id,
        picklist_name: row.picklist_name,
        picklist_values: []
      });
    }
    const agg = map.get(key);
    if (row.order_detail_id !== null) {
      // Push event-level data
      agg.picklist_values.push({
        id: row.id,
        picklist_option: row.picklist_option,
      });
    }
  });
  return Array.from(map.values());
}