const { sendErrorResponse, sendResponse } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getTimezone, getSystemTime } = require("../../functions/getTimezone.js");

// Create Employee
module.exports.createEmployee = async (req, res) => {
  try {
    const {
      employee_id_external,
      person_id_external,
      recruitment_date,
      joining_date,
      primary_employee = 2,
    } = req.body;

    // Required fields check
    if (
      !employee_id_external ||
      !person_id_external ||
      !recruitment_date ||
      !joining_date
    ) {
      const errorMessage = "employee_id_external, person_id_external, recruitment_date, joining_date are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // duplicate record check
    const existing = await performQuery(
      `SELECT * FROM ${tables.emp_employee} WHERE employee_id_external = ?`,
      [employee_id_external]
    );
    if (existing.length > 0) {
      const errorMessage = "Employee with the same employee_id_external already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert query
    await performQuery(`INSERT INTO ${tables.emp_employee} SET ?`, {
      employee_id_external,
      person_id_external,
      recruitment_date,
      joining_date,
      primary_employee,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    return sendResponse(res, {}, "Employee created successfully", 200);
  } catch (error) {
    return sendErrorResponse(
      res,
      error,
      "Error while fetching Employye records"
    );
  }
};

// Read Employee
module.exports.getEmployee = async (req, res) => {
  try {
    const { id, employee_id_external, person_id, person_id_external } =
      req.query;

    let query = `
      SELECT 
        emp.*,
        person.person_id_external, person.display_name AS person_display_name
      FROM ${tables.emp_employee} AS emp
      LEFT JOIN ${tables.per_person} AS person ON emp.person_id_external = person.person_id_external
      AND person.is_deleted = 2
      WHERE emp.is_deleted = 2
    `;
    if (id) query += ` AND emp.id = ${id}`;
    if (employee_id_external)
      query += ` AND dep.employee_id_external = '${employee_id_external}'`;
    if (person_id) query += ` AND emp.person_id = ${person_id}`;
    if (person_id_external)
      query += ` AND person.person_id_external = '${person_id_external}'`;

    query += ` ORDER BY emp.id DESC`;

    const employee = await performQuery(query);
    return sendResponse(res, employee, "Employee fetched successfully", 200);
  } catch (error) {
    console.error("Error while fetching Employee:", error);
    return sendErrorResponse(res, error, "Error while fetching Employee");
  }
};

// Update Employee
module.exports.updateEmployee = async (req, res) => {
  try {
    const {
      id,
      recruitment_date,
      joining_date,
      primary_employee = 2,
    } = req.body;

    // Required fields check
    if (
      !id ||
      !recruitment_date ||
      !joining_date
    ) {
      const errorMessage = "id, recruitment_date, joining_date are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check Record Exists
    const existing = await performQuery(
      `SELECT * FROM ${tables.emp_employee} WHERE id = ? AND is_deleted = 2`,
      [id]
    );

    if (!existing || existing.length === 0) {
      const errorMessage = "Position record not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const oldRecord = existing[0];

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Prepare update object
    const updatedData = {
      recruitment_date: recruitment_date || oldRecord.recruitment_date,
      joining_date: joining_date || oldRecord.joining_date,
      primary_employee: primary_employee || oldRecord.primary_employee,
      updated_at: currentTime,
      updated_by: req?.user?.id || null,
    };

    // Update query
    await performQuery(`UPDATE ${tables.emp_employee} SET ? WHERE id = ?`, [
      updatedData,
      id,
    ]);

    return sendResponse(res, {}, "Employee updated successfully", 200);
  } catch (error) {
    console.error("Error While Updating Employee: ", error);
    return sendErrorResponse(res, error, "Error while updating Employee");
  }
};
