const {
  sendErrorResponse,
  sendResponse,
} = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");


// CREATE Division External
module.exports.createDivision = async (req, res) => {
  try {
    const {
      division_id_external,
      business_unit_id,
      effective_start_date,
      description,
      description_arabic
    } = req.body;

    if (!division_id_external || !business_unit_id || !effective_start_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check business_unit_id if exisit in business_unit tabel
    const isBusinessUnitID = await performQuery(
      `SELECT * FROM ${tables.org_business_unit} WHERE id = ? AND is_deleted = 2`,
      [business_unit_id]
    );

    if (isBusinessUnitID.length === 0) {
      const errorMessage = "Business Unit ID Not Found against Business Unit";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check for duplicate division_id_external
    const existingDivision = await performQuery(
      `SELECT id FROM ${tables.org_division} WHERE division_id_external = ? AND is_deleted = 2`,
      [division_id_external]
    );
    if (existingDivision && existingDivision.length > 0) {
      const errorMessage = "Division ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert Record
    await performQuery(`INSERT INTO ${tables.org_division} SET ?`, {
      division_id_external,
      business_unit_id,
      effective_start_date,
      description,
      description_arabic,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    return sendResponse(res, {}, "Division External created successfully", 201);

  } catch (error) {
    console.error("Error while creating Division External:", error);
    return sendErrorResponse(res, error, "Error while creating Division External");
  }
};


// READ Division(s)
module.exports.getDivision = async (req, res) => {
  try {
    const { id, division_id_external, business_unit_id, bu_id_external, entity_id, entity_id_external } = req.query;

    let query = `
      SELECT 
        division.*, 
        bu.bu_id_external, bu.description AS business_unit_description, 
        le.entity_id_external, le.name, le.org_unit_code, le.org_registration_date
      FROM ${tables.org_division} division
      LEFT JOIN ${tables.org_business_unit} bu ON division.business_unit_id = bu.id AND bu.is_deleted = 2
      LEFT JOIN ${tables.org_legal_entity} le ON bu.entity_id = le.id AND le.is_deleted = 2
      WHERE division.is_deleted = 2
    `;

    if (id) query += ` AND division.id = ${id}`;
    if (division_id_external) query += ` AND division.division_id_external = '${division_id_external}'`;
    if (business_unit_id) query += ` AND division.business_unit_id = ${business_unit_id}`;
    if (bu_id_external) query += ` AND bu.bu_id_external = '${bu_id_external}'`;
    if (entity_id) query += ` AND le.id = ${entity_id}`;
    if (entity_id_external) query += ` AND le.entity_id_external = '${entity_id_external}'`;

    query += ` ORDER BY division.id DESC`;

    const divisions = await performQuery(query);
    return sendResponse(res, divisions, "Division(s) fetched successfully", 200);

  } catch (error) {
    console.error("Error while fetching Division(s):", error);
    return sendErrorResponse(res, error, "Error while fetching Division(s)");
  }
};


// UPDATE Division External
module.exports.updateDivision = async (req, res) => {
  try {
    const {
      id,
      division_id_external,
      business_unit_id,
      effective_start_date,
      description,
      description_arabic,
      status,
    } = req.body;

    if (!id || !division_id_external || !business_unit_id) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if record exists
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.org_division} WHERE id = ? AND is_deleted = 2`,
      [id]
    );
    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Division record not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    const oldRecord = existingRecord[0];

    // Check for duplicate division_id_external (excluding current record)
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_division} WHERE division_id_external = ? AND id <> ? AND is_deleted = 2`,
      [division_id_external, id]
    );
    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Division ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update record
    await performQuery(`Update ${tables.org_division} SET ? WHERE id = ?`, [{
      effective_start_date: effective_start_date || oldRecord.effective_start_date,
      description: description || oldRecord.description,
      description_arabic: description_arabic || oldRecord.description_arabic,
      status: status || oldRecord.status,
      is_deleted: 2,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    }, id]);

    return sendResponse(res, {}, "Division External updated successfully", 200);

  } catch (error) {
    console.error("Error while updating Division External:", error);
    return sendErrorResponse(res, error, "Error while updating Division External");
  }
};


// DELETE Division External (Soft Delete)
// module.exports.deleteDivision = async (req, res) => {
//   try {
//     const { id } = req.query;

//     if (!id) {
//       const errorMessage = "Division ID is required";
//       return sendErrorResponse(res, errorMessage, errorMessage);
//     }

//     // Get System Time
//     const systemTime = await getSystemTime();
//     const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

//     // Soft delete
//     await performQuery(
//       `UPDATE ${tables.org_division} SET ? WHERE id = ?`,
//       [
//         {
//           is_deleted: 1,
//           updated_at: currentTime,
//           updated_by: req?.user?.id,
//         },
//         id,
//       ]
//     );

//     return sendResponse(res, {}, "Division External deleted successfully", 200);
//   } catch (error) {
//     console.error("Error while deleting Division External:", error);
//     return sendErrorResponse(res, error, "Error while deleting Division External");
//   }
// };
