const {
  sendErrorResponse,
  sendResponse,
} = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getTimezone, getSystemTime } = require("../../functions/getTimezone.js");

// CREATE ADDRESS
module.exports.createAddress = async (req, res) => {
  try {    
    const {
      person_id,
      address_type,
      address_line_1,
      address_line_2,
      address_line_3,
      locality,
      dependent_locality,
      administrative_area,
      sub_administrative_area,
      postal_code,
      country_code,
      sorting_code,
      language_code,
      is_primary = 2,
      notes
    } = req.body;

    // Required fields check
    if (
        !person_id || !address_type || !address_line_1 || 
        !locality || !administrative_area || !sub_administrative_area ||
        !country_code || !is_primary
    ) {
      const errorMessage = "Person ID, Address Type and Address Line 1 are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if person exists
    const personCheck = await performQuery(
      `SELECT * FROM ${tables.per_person} WHERE id = ? AND is_deleted = 2`,
      [person_id]
    );
    if (personCheck.length === 0) {
      const errorMessage = "Person not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Calculate completion %
    const total_fields = 14; 
    let filled_fields = 0;
    const checkFilled = (v) => { if (v) filled_fields++; };
    [
      person_id, 
      address_type, 
      address_line_1, 
      address_line_2, 
      address_line_3,
      locality, 
      dependent_locality, 
      administrative_area, 
      sub_administrative_area,
      postal_code, 
      country_code, 
      sorting_code, 
      language_code, 
      is_primary, notes
    ].forEach(checkFilled);

    const completion = Math.round((filled_fields / total_fields) * 100);
    const is_completed = completion === 100 ? 2 : 1; // 2 = completed, 1 = not completed

    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert Query
    const insertQuery = `INSERT INTO ${tables.per_address} SET ?`;
    const insertParams = {
      person_id : person_id,
      address_type : address_type,
      address_line_1 : address_line_1,
      address_line_2 : address_line_2,
      address_line_3 : address_line_3,
      locality : locality,
      dependent_locality : dependent_locality,
      administrative_area : administrative_area,
      sub_administrative_area : sub_administrative_area,
      postal_code : postal_code,
      country_code : country_code,
      sorting_code : sorting_code,
      language_code : language_code,
      is_primary : is_primary,
      completion : completion,
      is_completed : is_completed,
      notes : notes,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    };

    const insertRecord = await performQuery(insertQuery, insertParams);

    // If Current is Primary, make others non-primary (isPrimary = 2)
    if(is_primary == 1){
      await performQuery(
        `UPDATE ${tables.per_address} SET ? WHERE id <> ? AND person_id = ?`,
        [
            {
                is_primary: 2,
                updated_at: currentTime,
                updated_by: req?.user?.id,
            },
            insertRecord.insertId, person_id
        ]
      );
    }

    // Success response
    return sendResponse(res, {}, "Address created successfully", 201);

  } catch (error) {
    console.error(error);
    return sendErrorResponse(res, error, "Error while creating Address");
  }
};


// READ Address (by person_id or person_id_external)
module.exports.getAddress = async (req, res) => {
  try {
    const { person_id, person_id_external } = req.query;

    let query = `
      SELECT 
        pa.*, pp.display_name AS person_name, pp.person_id_external
      FROM ${tables.per_address} AS pa
      LEFT JOIN ${tables.per_person} AS pp ON pa.person_id = pp.id
      WHERE pa.is_deleted = 2
    `;

    if (person_id) {
      query += ` AND pa.person_id = ${person_id}`;
    } 
    else if (person_id_external) {
      query += ` AND pp.person_id_external = '${person_id_external}'`;
    }

    // Sort by most recent id
    query += ` ORDER BY pa.id DESC`;

    // Execute query
    const addressRecords = await performQuery(query);

    // Send response
    return sendResponse(res, addressRecords, "Address record(s) fetched successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while fetching Address records");
  }
};


// UPDATE ADDRESS
module.exports.updateAddress = async (req, res) => {
  try {
    const {
      id,
      address_type,
      address_line_1,
      address_line_2,
      address_line_3,
      locality,
      dependent_locality,
      administrative_area,
      sub_administrative_area,
      postal_code,
      country_code,
      sorting_code,
      language_code,
      is_primary,
      notes
    } = req.body;

    // Required field check (ID is required)
    if (
        !id || !address_type || !address_line_1 || 
        !locality || !administrative_area || !sub_administrative_area ||
        !country_code || !is_primary
    ) {
      const errorMessage = "ID, Address Type and Address Line 1 are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get existing record
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.per_address} WHERE id = ? AND is_deleted = 2`,
      [id]
    );
    if (existingRecord.length === 0) {
      const errorMessage = "Address record not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    const oldRecord = existingRecord[0];

    // Calculate completion %
    const total_fields = 15;
    let filled_fields = 0;
    const checkFilled = (v) => { if (v) filled_fields++; };
    [
      oldRecord.person_id,
      address_type || oldRecord.address_type,
      address_line_1 || oldRecord.address_line_1,
      address_line_2 || oldRecord.address_line_2,
      address_line_3 || oldRecord.address_line_3,
      locality || oldRecord.locality,
      dependent_locality || oldRecord.dependent_locality,
      administrative_area || oldRecord.administrative_area,
      sub_administrative_area || oldRecord.sub_administrative_area,
      postal_code || oldRecord.postal_code,
      country_code || oldRecord.country_code,
      sorting_code || oldRecord.sorting_code,
      language_code || oldRecord.language_code,
      is_primary || oldRecord.is_primary,
      notes || oldRecord.notes
    ].forEach(checkFilled);

    const completion = Math.round((filled_fields / total_fields) * 100);
    const is_completed = completion === 100 ? 2 : 1;

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Mark old record as deleted
    await performQuery(
      `UPDATE ${tables.per_address} SET ? WHERE id = ?`,
      [
        {
          is_deleted: 1,
          updated_at: currentTime,
          updated_by: req?.user?.id,
        },
        id
      ]
    );

    // Insert new updated record
    const insertQuery = `INSERT INTO ${tables.per_address} SET ?`;
    const insertParams = {
      person_id: oldRecord.person_id,
      address_type: address_type || oldRecord.address_type,
      address_line_1: address_line_1 || oldRecord.address_line_1,
      address_line_2: address_line_2 || oldRecord.address_line_2,
      address_line_3: address_line_3 || oldRecord.address_line_3,
      locality: locality || oldRecord.locality,
      dependent_locality: dependent_locality || oldRecord.dependent_locality,
      administrative_area: administrative_area || oldRecord.administrative_area,
      sub_administrative_area: sub_administrative_area || oldRecord.sub_administrative_area,
      postal_code: postal_code || oldRecord.postal_code,
      country_code: country_code || oldRecord.country_code,
      sorting_code: sorting_code || oldRecord.sorting_code,
      language_code: language_code || oldRecord.language_code,
      is_primary: is_primary || oldRecord.is_primary,
      completion,
      is_completed,
      is_deleted: 2,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
      notes: notes || oldRecord.notes
    };

    const insertRecord = await performQuery(insertQuery, insertParams);

    // If Current is Primary, make others non-primary (isPrimary = 2)
    if(is_primary == 1){
      await performQuery(
        `UPDATE ${tables.per_address} SET ? WHERE id <> ? AND person_id = ?`,
        [
            {
                is_primary: 2,
                updated_at: currentTime,
                updated_by: req?.user?.id,
            },
            insertRecord.insertId, oldRecord.person_id
        ]
      );
    }

    // Success response
    return sendResponse(res, {}, "Address updated successfully", 200);

  } catch (error) {
    console.log(error);
    return sendErrorResponse(res, error, "Error while updating Address");
  }
};

// DELETE Address
module.exports.deleteAddress = async (req, res) => {
  try {
    const { id } = req.query;

    // Validate required field
    if (!id) {
      const errorMessage = "Address ID is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if record exists
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.per_address} WHERE id = ? AND is_deleted = 2`,
      [id]
    );

    if (existingRecord.length === 0) {
      const errorMessage = "Address record not found or already deleted";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get current system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Soft delete the record
    const deleteQuery = `UPDATE ${tables.per_address} SET ? WHERE id = ?`;
    const deleteParams = [
      {
        is_deleted: 1,
        updated_at: currentTime,
        updated_by: req?.user?.id
      },
      id
    ];

    await performQuery(deleteQuery, deleteParams);

    // Success response
    return sendResponse(res, {}, "Address deleted successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while deleting Address");
  }
};
