const { isUndefined } = require("lodash");
const { sendErrorResponse, sendResponse } = require("../../utils");
const moment = require("moment-timezone");
const fs = require('fs');
const path = require('path');
const { performQuery } = require("../../utils/db");
const { getSystemTime } = require("../../functions/getTimezone");
const { tables } = require("../../utils/tables");
const bcrypt = require("bcryptjs/dist/bcrypt");
const { generateOTP } = require("../../services/OTP");
const { sendMail } = require("../../services/MAIL");


// API to Create Personal Info Record --> POST /api/personal-info
module.exports.createPersonalInfoRecord = async (req, res) => {
    const uploadedFiles = [];
    try {
        // get Variable from req body
        const {
            salutation, first_name, mid_name, last_name, display_name,
            first_name_arabic, mid_name_arabic, last_name_arabic, display_name_arabic,
            date_of_birth, date_of_birth_hijri,
            gender, nationality, marital_status,
            native_prefered_language,
            email, email_type
        } = req.body;        

        // Validate the input
        if(
            !salutation || !first_name || !last_name || !display_name ||        
            !date_of_birth || !gender || !nationality ||
            !marital_status || !native_prefered_language ||
            !email || !email_type
        ){
            return sendErrorResponse(res, "Required fields are missing", "Validation Error", 400);
        }

        // Handle file uploads
        let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
        const profilePic = req.files?.attachment_1 ? req.files.attachment_1[0].filename : null;
        if (profilePic) uploadedFiles.push(profilePic);

        // check for duplicate email
        const existingEmail = await performQuery(
            `SELECT * FROM ${tables.per_email} WHERE email = ? AND is_deleted = 2`,
            [email]
        );
        if(existingEmail.length > 0){
            return sendErrorResponse(res, "Email address is already in use", "Duplicate Email", 409);
        }

        // calculate completion percentage
        const totalFields = 16; // total number of fields
        let filledFields = 0;

        const personalInfoTotalFields = 5; // total number of personal info fields
        let personalInfoFilledFields = 0;

        if (gender) personalInfoFilledFields++;
        if (nationality) personalInfoFilledFields++;
        if (marital_status) personalInfoFilledFields++;
        if (native_prefered_language) personalInfoFilledFields++;
        if (profilePic) personalInfoFilledFields++;
        const personalInfoCompletionPercentage = Math.round((personalInfoFilledFields / personalInfoTotalFields) * 10000)/100;
        const personalInfoIsCompleted = personalInfoCompletionPercentage === 100 ? 1 : 2;

        if (salutation) filledFields++;
        if (first_name) filledFields++;
        if (mid_name) filledFields++;
        if (last_name) filledFields++;
        if (display_name) filledFields++;
        if (date_of_birth) filledFields++;
        if (first_name_arabic) filledFields++;
        if (mid_name_arabic) filledFields++;
        if (last_name_arabic) filledFields++;
        if (display_name_arabic) filledFields++;
        if (date_of_birth_hijri) filledFields++;
        if (gender) filledFields++;
        if (nationality) filledFields++;
        if (marital_status) filledFields++;
        if (native_prefered_language) filledFields++;

        const completionPercentage = Math.round((filledFields / totalFields) * 10000)/100;
        const is_completed = completionPercentage === 100 ? 1 : 2;

        // Get current system time
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");
                
        // create File URLs
        const profilePicURL = profilePic ? `${baseURL}/${profilePic}` : null;

        // create person record
        const personRecord = await performQuery(
            `INSERT INTO ${tables.per_person} SET ?`,
            {
                salutation: salutation,
                first_name: first_name,
                mid_name: mid_name, 
                last_name: last_name,
                display_name: display_name,
                first_name_arabic: first_name_arabic,
                mid_name_arabic: mid_name_arabic,
                last_name_arabic: last_name_arabic,
                display_name_arabic: display_name_arabic,
                date_of_birth: date_of_birth,
                date_of_birth_hijri: date_of_birth_hijri,
                completion: completionPercentage,
                is_completed: is_completed,
                created_at: currentTime,
                created_by: req?.user?.id,
                updated_at: currentTime,
                updated_by: req?.user?.id
            }
        );

        const personId = personRecord.insertId;


        // create personal info record
        const personalInfoRecord = await performQuery(
            `INSERT INTO ${tables.per_personal_info} SET ?`,
            {
                person_id: personId,
                gender: gender,
                nationality: nationality,
                marital_status: marital_status,
                native_prefered_language: native_prefered_language,
                attachment_1: profilePicURL,
                completion: personalInfoCompletionPercentage,
                is_completed: personalInfoIsCompleted,
                created_at: currentTime,
                created_by: req?.user?.id,
                updated_at: currentTime,
                updated_by: req?.user?.id
            }
        );

        // create email record
        const emailRecord = await performQuery(
            `INSERT INTO ${tables.per_email} SET ?`,
            {
                person_id: personId,
                email: email,
                email_type: email_type,
                is_primary: 1,
                created_at: currentTime,
                created_by: req?.user?.id,
                updated_at: currentTime,
                updated_by: req?.user?.id
            }
        );

        const hashedPassword = await bcrypt.hash(`Applicant_${first_name}@${personId}`, 10); // Default password, should be changed later
        
        // Generate a one-time password (verification_otp) for email verification
        const verification_otp = generateOTP();

        // create login data record
        const loginDataRecord = await performQuery(
            `INSERT INTO ${tables.per_login_data} SET ?`,
            {
                person_id: personId,
                email_id: emailRecord.insertId,
                password: hashedPassword,
                verification_otp: verification_otp,
                verification_otp_at: currentTime,
                is_new: 1,
                created_at: currentTime,
                created_by: req?.user?.id,
                updated_at: currentTime,
                updated_by: req?.user?.id
            }
        );

        // If the employee is created successfully, send a response
        const emailBody = `
            <!DOCTYPE html>
            <html>
                <head>
                    <title>Applicant Account Created Successfully</title>
                </head>
                <body style="font-family: Arial, sans-serif; background-color: #f4f4f4; padding: 20px;">
                    <div style="max-width: 600px; margin: 0 auto; background-color: #ffffff; padding: 30px; border-radius: 10px;">
                    <h1 style="color: #333333;">Dear ${salutation} ${first_name} ${!mid_name ? '' : mid_name} ${last_name},</h1>
                    <p style="color: #666666;">Your account has been created successfully.</p>
                    <p style="color: #666666;">Please use the following credentials to log in:</p>
                    <p style="color: #666666;">Email: <b>${email}</b></p>
                    <p style="color: #666666;">Password: <b>Applicant_${first_name}@${personId}</b></p>
                    <p style="color: #666666;">Verification OTP: <b>${verification_otp}</b></p>
                    <p style="color: #666666; font-size: 12px;">If you did not request this code, please ignore this email.</p>
                    </div>
                </body>
            </html>
        `;
        
        // Send the verification_otp to the employee's email address
        await sendMail(email, "Verify your email address", emailBody);

        return sendResponse(res, {person_id: personId, display_name: display_name}, "Personal info record created successfully", 201);

    } catch (error) {
        console.log("Error in createPersonalInfoRecord:", error);
        return sendErrorResponse(res, error, "Failed to create personal info record", 500);
    }
}

// API to Update Personal Info Record --> PUT /api/personal-info
module.exports.updatePersonalInfoRecord = async (req, res) => {
    const uploadedFiles = [];
    try {
        // get Variable from req body
        const {
            id, first_name_arabic, mid_name_arabic, last_name_arabic, display_name_arabic,
            date_of_birth, date_of_birth_hijri,
            gender, nationality, marital_status,
            native_prefered_language
        } = req.body;        

        // Validate the input
        if(
            !id || !gender || !nationality ||
            !marital_status || !native_prefered_language
        ){
            return sendErrorResponse(res, "Required fields are missing", "Validation Error", 400);
        }

        // Handle file uploads
        let baseURL = `${req.protocol}://${req.get("host")}/uploads`;
        const profilePic = req.files?.attachment_1 ? req.files.attachment_1[0].filename : null;
        if (profilePic) uploadedFiles.push(profilePic);

        // check of data exists
        const personalInfoData = await performQuery(`
            SELECT 
                pp.id, pp.person_id_external, pp.salutation, pp.first_name, pp.mid_name, pp.last_name, pp.display_name,
                pp.first_name_arabic, pp.mid_name_arabic, pp.last_name_arabic, pp.display_name_arabic,
                pp.date_of_birth, pp.date_of_birth_hijri, 
                ppi.id AS personal_info_id, ppi.gender, ppi.nationality, ppi.marital_status, ppi.native_prefered_language, ppi.attachment_1,
                pe.id AS person_email_id, pe.email 
            FROM ${tables.per_person} AS pp
            Left JOIN ${tables.per_personal_info} AS ppi ON (pp.id = ppi.person_id AND ppi.is_deleted = 2)
            left JOIN ${tables.per_email} AS pe ON (pp.id = pe.person_id AND pe.is_primary = 1 AND pe.is_deleted = 2)
            WHERE pp.id = ? AND pp.is_deleted = 2`,
            [id]
        );
        if(personalInfoData.length === 0){
            return sendErrorResponse(res, "Personal info record not found", "Not Found", 404);
        }

        const personData = personalInfoData[0];

        // calculate completion percentage
        const totalFields = 16; // total number of fields
        let filledFields = 0;

        const personalInfoTotalFields = 5; // total number of personal info fields
        let personalInfoFilledFields = 0;

        if (personData?.gender || gender) personalInfoFilledFields++;
        if (personData?.nationality || nationality) personalInfoFilledFields++;
        if (personData?.marital_status || marital_status) personalInfoFilledFields++;
        if (personData?.native_prefered_language || native_prefered_language) personalInfoFilledFields++;
        if (personData.attachment_1 || profilePic) personalInfoFilledFields++;

        const personalInfoCompletionPercentage = Math.round((personalInfoFilledFields / personalInfoTotalFields) * 10000)/100;
        const personalInfoIsCompleted = personalInfoCompletionPercentage === 100 ? 1 : 2;

        if (personData?.salutation) filledFields++;
        if (personData?.first_name) filledFields++;
        if (personData?.mid_name) filledFields++;
        if (personData?.last_name) filledFields++;
        if (personData?.display_name) filledFields++;
        if (personData?.date_of_birth || date_of_birth) filledFields++;
        if (personData?.first_name_arabic || first_name_arabic) filledFields++;
        if (personData?.mid_name_arabic || mid_name_arabic) filledFields++;
        if (personData?.last_name_arabic || last_name_arabic) filledFields++;
        if (personData?.display_name_arabic || display_name_arabic) filledFields++;
        if (personData?.date_of_birth_hijri || date_of_birth_hijri) filledFields++;
        if (personData?.gender || gender) filledFields++;
        if (personData?.nationality || nationality) filledFields++;
        if (personData?.marital_status || marital_status) filledFields++;
        if (personData?.native_prefered_language || native_prefered_language) filledFields++;

        const completionPercentage = Math.round((filledFields / totalFields) * 10000)/100;
        const is_completed = completionPercentage === 100 ? 1 : 2;

        // Get current system time
        const systemTime = await getSystemTime();
        const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");
                
        // create File URLs
        const profilePicURL = profilePic ? `${baseURL}/${profilePic}` : null;


        // update person record
        const personRecord = await performQuery(
            `UPDATE ${tables.per_person} SET ? WHERE id = ?`,
            [
                {
                    first_name_arabic: first_name_arabic || personData.first_name_arabic,
                    mid_name_arabic: mid_name_arabic || personData.mid_name_arabic,
                    last_name_arabic: last_name_arabic || personData.last_name_arabic,
                    display_name_arabic: display_name_arabic || personData.display_name_arabic,
                    date_of_birth: date_of_birth || personData.date_of_birth,
                    date_of_birth_hijri: date_of_birth_hijri || personData.date_of_birth_hijri,
                    completion: completionPercentage,
                    is_completed: is_completed,
                    updated_at: currentTime,
                    updated_by: req?.user?.id
                },
            id]
    );

        const personId = id;
        const personalInfoId = personData.personal_info_id;

        if (personalInfoId) {
            // soft delete existing personal info record to make history
            const softDeletePersonalInfo = await performQuery(
                `UPDATE ${tables.per_personal_info} SET ? WHERE id = ?`,
                [
                    {
                        is_deleted: 1,
                        updated_at: currentTime,
                        updated_by: req?.user?.id
                    },
                    personalInfoId
                ]
            ); 
        }

        // Create personal info record
        const personalInfoRecord = await performQuery(
            `INSERT INTO ${tables.per_personal_info} SET ?`,
            {
                person_id: personId,
                gender: gender || personData.gender,
                nationality: nationality || personData.nationality,
                marital_status: marital_status || personData.marital_status,
                native_prefered_language: native_prefered_language || personData.native_prefered_language,
                attachment_1: profilePicURL || personData.attachment_1,
                completion: personalInfoCompletionPercentage,
                is_completed: personalInfoIsCompleted,
                created_at: currentTime,
                created_by: req?.user?.id,
                updated_at: currentTime,
                updated_by: req?.user?.id
            }
        );

        return sendResponse(res, [], "Personal info record Updated successfully", 201);

    } catch (error) {
        console.log("Error in updatePersonalInfoRecord:", error);
        return sendErrorResponse(res, error, "Failed to update personal info record", 500);
    }
}

// API to Get Personal Info Record --> GET /api/personal-info
module.exports.getPersonalInfoRecord = async (req, res) => {
    try {
        // get Variable from req query
        const { id, person_id_external } = req.query;

        // query
        let query = `
            SELECT 
                pp.id, pp.person_id_external, pp.salutation, pp.first_name, pp.mid_name, pp.last_name, pp.display_name,
                pp.first_name_arabic, pp.mid_name_arabic, pp.last_name_arabic, pp.display_name_arabic,
                pp.date_of_birth, pp.date_of_birth_hijri, pp.completion, pp.is_completed,
                ppi.id AS personal_info, ppi.gender, ppi.nationality, ppi.marital_status, ppi.native_prefered_language, ppi.attachment_1
            FROM ${tables.per_person} AS pp
            LEFT JOIN ${tables.per_personal_info} AS ppi ON (pp.id = ppi.person_id AND ppi.is_deleted = 2)
            WHERE pp.is_deleted = 2 
        `

        if (!isUndefined(id)) {
            query += ` AND pp.id = ${id} `;
        }
        if (!isUndefined(person_id_external)) {
            query += ` AND pp.person_id_external = '${person_id_external}' `;
        }

        // sort data in decreasing order of id
        query += ` ORDER BY pp.id DESC `;
        
        const result = await performQuery(query, []);
        return sendResponse(res, result, "Personal info record retrieved successfully", 200);

    } catch (error) {
        console.log("Error in getPersonalInfoRecord:", error);
        return sendErrorResponse(res, error, "Failed to get personal info record", 500);
    }
}

// API to Delete Personal Info Record --> DELETE /api/personal-info
module.exports.deletePersonalInfoRecord = async (req, res) => {
    try {
        // get Variable from req query
        const { id } = req.query;

        // validate input
        if (isUndefined(id)) {
            return sendErrorResponse(res, "Personal info ID is required", "Validation Error", 400);
        }

        return sendErrorResponse(res, "Personal info deletion is not allowed", "Operation Not Allowed", 403);

    } catch (error) {
        console.log("Error in deletePersonalInfoRecord:", error);
        return sendErrorResponse(res, error, "Failed to delete personal info record", 500);
    }
}

