const { sendErrorResponse, sendResponse, } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// --> Create Time Profile Details
module.exports.createTimeProfileDetails = async (req, res) => {
    try {
    const {
      time_profile_id_external,
      time_type_id_external,
      assigned_quota,
      is_one_time,
      evaluated_on_runtime,
      external_code
    } = req.body;

    if (!time_profile_id_external || !time_type_id_external || !external_code || !is_one_time || !evaluated_on_runtime) {
      return sendErrorResponse(res, "Missing required fields!", "Missing required fields!");
    }

    // Duplicate check
    const existing = await performQuery(
      `SELECT * FROM ${tables.time_profile_details}
       WHERE time_profile_id_external = ?
       AND (time_type_id_external = ? OR external_code = ?)
       AND is_active = 1`,
      [time_profile_id_external, time_type_id_external, external_code]
    );
    if (existing.length > 0) {
      return sendErrorResponse(res, "Time Profile Detail already exists", "Time Profile Detail already exists");
    }

    // Check if time_type_id_external exists
    const timeTypeExists = await performQuery(
      `SELECT * FROM ${tables.time_type} WHERE time_type_id_external = ? AND is_active = 1`,
      [time_type_id_external]
    );
    if (timeTypeExists.length === 0) {
      return sendErrorResponse(res, "Time Type not found", "Time Type not found");
    }

    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    await performQuery(
      `INSERT INTO ${tables.time_profile_details} SET ?`,
      {
        sequence_no: 1,
        time_profile_id_external,
        time_type_id_external,
        assigned_quota: assigned_quota || null,
        is_one_time,
        evaluated_on_runtime,
        external_code,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Time Profile Detail created successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while creating Time Profile Detail");
  }
};

// --> Get Time Profile Details
module.exports.getTimeProfileDetails = async (req, res) => {
    try {
    const { id, time_profile_id_external } = req.query;

    let query = `
      SELECT tpd.*, tp.extrenal_name AS time_profile_name, tt.external_name AS time_type_name
      FROM ${tables.time_profile_details} tpd
      LEFT JOIN ${tables.time_profile} tp ON tpd.time_profile_id_external = tp.time_profile_id_external AND tp.is_active = 1
      LEFT JOIN ${tables.time_type} tt ON tpd.time_type_id_external = tt.time_type_id_external AND tt.is_active = 1
      WHERE tpd.is_active = 1
    `;

    if (id) query += ` AND tpd.id = ${id}`;
    if (time_profile_id_external)
      query += ` AND tpd.time_profile_id_external = '${time_profile_id_external}'`;

    query += ` ORDER BY tpd.id DESC`;
    const result = await performQuery(query);
    return sendResponse(res, result, "Time Profile Details fetched successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while fetching Time Profile Details");
  }
};

// --> Update Time Profile Details
module.exports.updateTimeProfileDetails = async (req, res) => {
    try {
    const { id, time_type_id_external, external_code, assigned_quota, is_one_time, evaluated_on_runtime } = req.body;

    if (!id || !time_type_id_external || !external_code || !is_one_time || !evaluated_on_runtime) {
      return sendErrorResponse(res, "Missing required fields!", "Missing required fields!");
    }

    const existing = await performQuery(
      `SELECT * FROM ${tables.time_profile_details} WHERE id = ?`,
      [id]
    );
    if (existing.length === 0) {
      return sendErrorResponse(res, "Time Profile Detail not found", "Time Profile Detail not found");
    }
    // Check if time_type_id_external exists
    const timeTypeExists = await performQuery(
      `SELECT * FROM ${tables.time_type} WHERE time_type_id_external = ? AND is_active = 1`,
      [time_type_id_external]
    );
    if (timeTypeExists.length === 0) {
      return sendErrorResponse(res, "Time Type not found", "Time Type not found");
    }
    // check for duplicate time_type_id_external or external_code
    const duplicateCheck = await performQuery(
      `SELECT * FROM ${tables.time_profile_details} WHERE (time_type_id_external = ? OR external_code = ?) AND is_active = 1 AND id != ?`,
      [time_type_id_external, external_code, id]
    );
    if (duplicateCheck.length > 0) {
      return sendErrorResponse(res, "Time Profile Detail with given Time Type or External Code already exists", "Time Profile Detail with given Time Type or External Code already exists");
    }

    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    await performQuery(
      `UPDATE ${tables.time_profile_details} SET ? WHERE id = ?`,
      [{ is_active: 2, updated_by: req?.user?.id, updated_at: currentTime }, id]
    );

    await performQuery(
      `INSERT INTO ${tables.time_profile_details} SET ?`,
      {
        sequence_no: Number(existing[0].sequence_no) + 1,
        time_profile_id_external: existing[0].time_profile_id_external,
        time_type_id_external: time_type_id_external || existing[0].time_type_id_external,
        external_code,
        assigned_quota: assigned_quota || null,
        is_one_time,
        evaluated_on_runtime,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Time Profile Detail updated successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while updating Time Profile Detail");
  }
};