const { sendErrorResponse, sendResponse, } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");

// --> Create Time Type
module.exports.createTimeType = async (req, res) => {
    
    try {
    const {
      time_type_id_external,
      external_name,
      unit,
      country,
      classification,
      permitted_fractions_unit_day,
      permitted_fractions_unit_hour
    } = req.body;

    if (
      !time_type_id_external ||
      !external_name ||
      !unit ||
      !country ||
      !classification
    ) {
      const errorMessage = "Missing required fields!";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Duplicate check
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.time_type}
       WHERE time_type_id_external = ? AND is_active = 1`,
      [time_type_id_external]
    );
    if (existingRecord.length > 0) {
      return sendErrorResponse(
        res,
        "Time Type already exists",
        "Time Type already exists"
      );
    }

    // Picklist check (classification)
    const checkClassification = await performQuery(
      `SELECT * FROM ${tables.picklist_master}
       WHERE picklist_id = 22 AND id = ? AND is_deleted = 2`,
      [classification]
    );
    if (checkClassification.length === 0) {
      return sendErrorResponse(res, "Invalid classification", "Invalid classification");
    }

    // Picklist check (unit)
    const checkUnit = await performQuery(
      `SELECT * FROM ${tables.picklist_master}
        WHERE picklist_id = 25 AND id = ? AND is_deleted = 2`,  
      [unit]
    );
    if (checkUnit.length === 0) {
      return sendErrorResponse(res, "Invalid unit", "Invalid unit");
    }

    if(!permitted_fractions_unit_day || permitted_fractions_unit_day > 0){
      // Picklist check (permitted_fractions_unit_day)
      const checkPermittedFractionsUnitDay = await performQuery(
        `SELECT * FROM ${tables.picklist_master}
          WHERE picklist_id = 23 AND id = ? AND is_deleted = 2`,  
        [permitted_fractions_unit_day]
      );
      if (checkPermittedFractionsUnitDay.length === 0) {
        return sendErrorResponse(res, "Invalid permitted_fractions_unit_day", "Invalid permitted_fractions_unit_day");
      }
    }

    if(!permitted_fractions_unit_hour || permitted_fractions_unit_hour > 0){
      // Picklist check (permitted_fractions_unit_hour)
      const checkPermittedFractionsUnitHour = await performQuery(
        `SELECT * FROM ${tables.picklist_master}
          WHERE picklist_id = 24 AND id = ? AND is_deleted = 2`,  
        [permitted_fractions_unit_hour]
      );
      if (checkPermittedFractionsUnitHour.length === 0) {
        return sendErrorResponse(res, "Invalid permitted_fractions_unit_hour", "Invalid permitted_fractions_unit_hour");
      }
    }

    // Get current system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    await performQuery(
      `INSERT INTO ${tables.time_type} SET ?`,
      {
        sequence_no: 1,
        time_type_id_external,
        external_name,
        unit,
        country,
        classification,
        permitted_fractions_unit_day,
        permitted_fractions_unit_hour,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Time Type created successfully", 200);

  } catch (error) {
    console.error("Error While Creating Time Type", error);
    return sendErrorResponse(res, error, "Error while creating Time Type");
  }
};

// --> Get Time Type
module.exports.getTimeType = async (req, res) => {
     try {
    const { id, time_type_id_external } = req.query;

    let query = `
      SELECT
        tt.id,
        tt.sequence_no,
        tt.time_type_id_external,
        tt.external_name,
        tt.unit, unit_pm.picklist_option AS unit_value,
        tt.country,
        tt.classification,
        pm.picklist_option AS classification_value,
        tt.permitted_fractions_unit_day, pfud_pm.picklist_option AS permitted_fractions_unit_day_value,
        tt.permitted_fractions_unit_hour, pfuhr_pm.picklist_option AS permitted_fractions_unit_hour_value
      FROM ${tables.time_type} tt
      LEFT JOIN ${tables.picklist_master} pm
        ON tt.classification = pm.id
        AND pm.picklist_id = 22
        AND pm.is_deleted = 2
      LEFT JOIN ${tables.picklist_master} unit_pm
        ON tt.unit = unit_pm.id
        AND unit_pm.picklist_id = 25
        AND unit_pm.is_deleted = 2
      LEFT JOIN ${tables.picklist_master} pfud_pm
        ON tt.permitted_fractions_unit_day = pfud_pm.id
        AND pfud_pm.picklist_id = 23
        AND pfud_pm.is_deleted = 2
      LEFT JOIN ${tables.picklist_master} pfuhr_pm
        ON tt.permitted_fractions_unit_hour = pfuhr_pm.id
        AND pfuhr_pm.picklist_id = 24
        AND pfuhr_pm.is_deleted = 2
      WHERE tt.is_active = 1
    `;

    if (id) query += ` AND tt.id = ${id}`;
    if (time_type_id_external)
      query += ` AND tt.time_type_id_external = '${time_type_id_external}'`;

    query += ` ORDER BY tt.id DESC`;

    const result = await performQuery(query);
    return sendResponse(res, result, "Time Type fetched successfully", 200);

  } catch (error) {
    console.error("Error While Fetching Time Type", error);
    return sendErrorResponse(res, error, "Error While Fetching Time Type");
  }
};

// --> Update Time Type
module.exports.updateTimeType = async (req, res) => {
    try {
    const {
      id,
      external_name,
      unit,
      country,
      classification,
      permitted_fractions_unit_day,
      permitted_fractions_unit_hour
    } = req.body;

    if (
      !id ||
      !external_name ||
      !unit ||
      !country ||
      !classification
    ) {
      const errorMessage = "Missing required fields!";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.time_type} WHERE id = ?`,
      [id]
    );
    if (existingRecord.length === 0) {
      return sendErrorResponse(res, "Time Type not found", "Time Type not found");
    }

    // Picklist check (classification)
    const checkClassification = await performQuery(
      `SELECT * FROM ${tables.picklist_master}
       WHERE picklist_id = 22 AND id = ? AND is_deleted = 2`,
      [classification]
    );
    if (checkClassification.length === 0) {
      return sendErrorResponse(res, "Invalid classification", "Invalid classification");
    }

    // Picklist check (unit)
    const checkUnit = await performQuery(
      `SELECT * FROM ${tables.picklist_master}
        WHERE picklist_id = 25 AND id = ? AND is_deleted = 2`,  
      [unit]
    );
    if (checkUnit.length === 0) {
      return sendErrorResponse(res, "Invalid unit", "Invalid unit");
    }

    if(!permitted_fractions_unit_day || permitted_fractions_unit_day > 0){
      // Picklist check (permitted_fractions_unit_day)
      const checkPermittedFractionsUnitDay = await performQuery(
        `SELECT * FROM ${tables.picklist_master}
          WHERE picklist_id = 23 AND id = ? AND is_deleted = 2`,  
        [permitted_fractions_unit_day]
      );
      if (checkPermittedFractionsUnitDay.length === 0) {
        return sendErrorResponse(res, "Invalid permitted_fractions_unit_day", "Invalid permitted_fractions_unit_day");
      }
    }

    if(!permitted_fractions_unit_hour || permitted_fractions_unit_hour > 0){
      // Picklist check (permitted_fractions_unit_hour)
      const checkPermittedFractionsUnitHour = await performQuery(
        `SELECT * FROM ${tables.picklist_master}
          WHERE picklist_id = 24 AND id = ? AND is_deleted = 2`,  
        [permitted_fractions_unit_hour]
      );
      if (checkPermittedFractionsUnitHour.length === 0) {
        return sendErrorResponse(res, "Invalid permitted_fractions_unit_hour", "Invalid permitted_fractions_unit_hour");
      }
    }

    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Deactivate old record
    await performQuery(
      `UPDATE ${tables.time_type} SET ? WHERE id = ?`,
      [{
        is_active: 2,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }, id]
    );

    // Insert new record
    await performQuery(
      `INSERT INTO ${tables.time_type} SET ?`,
      {
        sequence_no: Number(existingRecord[0].sequence_no) + 1,
        time_type_id_external: existingRecord[0].time_type_id_external,
        external_name,
        unit,
        country,
        classification,
        permitted_fractions_unit_day,
        permitted_fractions_unit_hour,
        created_by: req?.user?.id,
        created_at: currentTime,
        updated_by: req?.user?.id,
        updated_at: currentTime
      }
    );

    return sendResponse(res, {}, "Time Type updated successfully", 200);

  } catch (error) {
    console.error("Error While Updating Time Type", error);
    return sendErrorResponse(res, error, "Error while updating Time Type");
  }
};


