const { sendErrorResponse, sendResponse } = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const {
  getTimezone,
  getSystemTime,
} = require("../../functions/getTimezone.js");
const { time } = require("node:console");

// Create Employee
module.exports.createEmployee = async (req, res) => {
  try {
    const {
      employee_id_external,
      person_id_external,
      recruitment_date,
      joining_date,
      primary_employee = 2,
    } = req.body;

    // Required fields check
    if (
      !employee_id_external ||
      !person_id_external ||
      !recruitment_date ||
      !joining_date
    ) {
      const errorMessage =
        "employee_id_external, person_id_external, recruitment_date, joining_date are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // duplicate record check
    const existing = await performQuery(
      `SELECT * FROM ${tables.emp_employee} WHERE employee_id_external = ?`,
      [employee_id_external]
    );
    if (existing.length > 0) {
      const errorMessage =
        "Employee with the same employee_id_external already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert query
    await performQuery(`INSERT INTO ${tables.emp_employee} SET ?`, {
      employee_id_external,
      person_id_external,
      recruitment_date,
      joining_date,
      primary_employee,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    return sendResponse(res, {}, "Employee created successfully", 200);
  } catch (error) {
    return sendErrorResponse(
      res,
      error,
      "Error while fetching Employye records"
    );
  }
};

// Read Employee
module.exports.getEmployee = async (req, res) => {
  try {
    const { id, employee_id_external, person_id, person_id_external } =
      req.query;

    let query = `
      SELECT 
        emp.*,
        person.person_id_external, person.id AS person_id ,person.display_name AS person_display_name
      FROM ${tables.emp_employee} AS emp
      LEFT JOIN ${tables.per_person} AS person ON emp.person_id_external = person.person_id_external
      AND person.is_deleted = 2
      WHERE emp.is_deleted = 2
    `;
    if (id) query += ` AND emp.id = ${id}`;
    if (employee_id_external)
      query += ` AND dep.employee_id_external = '${employee_id_external}'`;
    if (person_id) query += ` AND emp.person_id = ${person_id}`;
    if (person_id_external)
      query += ` AND person.person_id_external = '${person_id_external}'`;

    query += ` ORDER BY emp.id DESC`;

    const employee = await performQuery(query);
    return sendResponse(res, employee, "Employee fetched successfully", 200);
  } catch (error) {
    console.error("Error while fetching Employee:", error);
    return sendErrorResponse(res, error, "Error while fetching Employee");
  }
};

// Update Employee
module.exports.updateEmployee = async (req, res) => {
  try {
    const {
      id,
      recruitment_date,
      joining_date,
      primary_employee = 2,
    } = req.body;

    // Required fields check
    if (!id || !recruitment_date || !joining_date) {
      const errorMessage = "id, recruitment_date, joining_date are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check Record Exists
    const existing = await performQuery(
      `SELECT * FROM ${tables.emp_employee} WHERE id = ? AND is_deleted = 2`,
      [id]
    );

    if (!existing || existing.length === 0) {
      const errorMessage = "Position record not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    const oldRecord = existing[0];

    // System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Prepare update object
    const updatedData = {
      employee_id_external: oldRecord.employee_id_external,
      sequence_no: Number(oldRecord.sequence_no) + 1,
      person_id_external: oldRecord.person_id_external,
      recruitment_date: recruitment_date || oldRecord.recruitment_date,
      joining_date: joining_date || oldRecord.joining_date,
      primary_employee: primary_employee || oldRecord.primary_employee,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id || null,
    };

    // Update query to maintain history
    await performQuery(`UPDATE ${tables.emp_employee} SET ? WHERE id = ?`, [{
      is_deleted: 1,
      updated_at: currentTime,
      updated_by: req?.user?.id || null,
    }, id]);

    // Insert new record with updated data
    await performQuery(`INSERT INTO ${tables.emp_employee} SET ?`, updatedData);

    return sendResponse(res, {}, "Employee updated successfully", 200);
  } catch (error) {
    console.error("Error While Updating Employee: ", error);
    return sendErrorResponse(res, error, "Error while updating Employee");
  }
};

// Get Employee Info (All Details)
module.exports.getEmployeeInfo = async (req, res) => {
  try {
    const { employee_id_external } = req.query;

    // Validation
    if (!employee_id_external) {
      const errorMessage = "employee_id_external is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    let personQuery = `
      SELECT
        pp.id, pp.person_id_external, 
        pp.salutation, pp.first_name, pp.mid_name, pp.last_name, pp.display_name,
        pp.first_name_arabic, pp.mid_name_arabic, pp.last_name_arabic, pp.display_name_arabic,
        pp.date_of_birth, pp.date_of_birth_hijri, 
        pp.completion, pp.is_completed,
        ppi.id AS personal_info_id, 
        ppi.gender, ppi.nationality, ppi.marital_status, 
        ppi.native_prefered_language, ppi.attachment_1,

        pni.id AS national_id_info_id, pni.country AS national_id_country,
        pni.national_id_no, pni.id_type,
        pni.issue_date AS national_id_issue_date, pni.issue_date_hijri AS national_id_issue_date_hijri, 
        pni.expiry_date AS national_id_expiry_date, pni.expiry_date_hijri AS national_id_expiry_date_hijri,
        pni.attachment_1 AS national_id_attachment_1, 
        pni.attachment_2 AS national_id_attachment_2,
        pni.completion AS national_id_completion, 
        pni.is_completed AS national_id_is_completed,

        pdl.id AS driving_license_info_id, pdl.country AS driving_license_country,
        pdl.license_no, 
        pdl.issue_date AS driving_license_issue_date, pdl.issue_date_hijri AS driving_license_issue_date_hijri,
        pdl.expiry_date AS driving_license_expiry_date, pdl.expiry_date_hijri AS driving_license_expiry_date_hijri,
        pdl.attachment_1 AS driving_license_attachment_1, 
        pdl.attachment_2 AS driving_license_attachment_2,
        pdl.completion AS driving_license_completion, 
        pdl.is_completed AS driving_license_is_completed,

        pvi.id AS visa_iqama_info_id, pvi.country AS visa_iqama_country,
        pvi.document_no, pvi.document_type AS visa_iqama_document_type,
        pvi.boarder_no, 
        pvi.issue_date AS visa_iqama_issue_date, pvi.issue_date_hijri AS visa_iqama_issue_date_hijri,
        pvi.issue_place AS visa_iqama_issue_place, pvi.issue_place AS visa_iqama_issue_place,
        pvi.issuing_authority AS visa_iqama_issuing_authority, pvi.validated AS visa_iqama_validated,
        pvi.expiry_date AS visa_iqama_expiry_date, pvi.expiry_date_hijri AS visa_iqama_expiry_date_hijri,
        pvi.attachment_1 AS visa_iqama_attachment_1, 
        pvi.attachment_2 AS visa_iqama_attachment_2,
        pvi.notes AS visa_iqama_notes,
        pvi.completion AS visa_iqama_completion, 
        pvi.is_completed AS visa_iqama_is_completed,

        pppt.id AS passport_info_id, pppt.country AS passport_country,
        pppt.passport_no, 
        pppt.issue_date AS passport_issue_date, pppt.issue_date_hijri AS passport_issue_date_hijri,
        pppt.issue_place AS passport_issue_place, pppt.issuing_authority AS passport_issuing_authority,
        pppt.validated AS passport_validated,
        pppt.expiry_date AS passport_expiry_date, pppt.expiry_date_hijri AS passport_expiry_date_hijri,
        pppt.attachment_1 AS passport_attachment_1, 
        pppt.attachment_2 AS passport_attachment_2,
        pppt.completion AS passport_completion, 
        pppt.is_completed AS passport_is_completed,
        pppt.notes AS passport_notes,

        pe.id AS email_info_id, pe.email_type, 
        pe.email, pe.email_type, pe.is_primary AS email_is_primary,
        pe.is_verified AS email_is_verified,

        pcn.id AS contact_info_id, pcn.country_code,
        pcn.contact_no, pcn.contact_type, pcn.is_primary AS contact_is_primary,

        pec.id AS emergency_contact_info_id,
        pec.name AS emergency_contact_name,
        pec.relation AS emergency_contact_relation,
        pec.phone AS emergency_contact_phone,
        pec.alternate_phone AS emergency_contact_alternate_phone,
        pec.address_line_1 AS emergency_contact_address_line_1,
        pec.address_line_2 AS emergency_contact_address_line_2,
        pec.address_line_3 AS emergency_contact_address_line_3,
        pec.locality AS emergency_contact_locality,
        pec.dependent_locality AS emergency_contact_dependent_locality,
        pec.administrative_area AS emergency_contact_administrative_area,
        pec.sub_administrative_area AS emergency_contact_sub_administrative_area,
        pec.postal_code AS emergency_contact_postal_code,
        pec.country_code AS emergency_contact_country_code,
        pec.notes AS emergency_contact_notes,
        pec.same_address AS emergency_contact_same_address,
        pec.completion AS emergency_contact_completion, 
        pec.is_completed AS emergency_contact_is_completed,

        pad.id AS address_info_id,
        pad.address_type, 
        pad.address_line_1, pad.address_line_2, pad.address_line_3,
        pad.locality, pad.dependent_locality, 
        pad.administrative_area, pad.sub_administrative_area,
        pad.postal_code, pad.country_code,
        pad.sorting_code, pad.language_code,
        pad.is_primary AS address_is_primary,
        pad.completion AS address_completion, 
        pad.is_completed AS address_is_completed,

        ped.id AS educational_info_id, 
        ped.course_type AS educational_course_type, 
        ct.picklist_option AS educational_course_type_value,
        ped.document_name AS educational_document_name, 
        ped.degree_title AS educational_degree_title, 
        dt.picklist_option AS educational_degree_title_value,
        ped.grade AS educational_grade,
        ped.institute_name AS educational_institute_name,
        ped.major_subject AS educational_major_subject,
        ped.document_no AS educational_document_no, 
        ped.start_date AS educational_start_date,
        ped.end_date AS educational_end_date,
        ped.status AS educational_status,
        ped.attachment_1 AS educational_attachment_1, 
        ped.attachment_2 AS educational_attachment_2,
        ped.completion AS educational_completion, 
        ped.is_completed AS educational_is_completed,

        ppibd.id AS payment_info_id,
        ppibd.effective_start_date AS payment_effective_start_date,
        ppibd.pay_type AS payment_pay_type,
        pt.picklist_option AS payment_pay_type_value,
        ppibd.customer_type_external_code AS payment_customer_type_external_code,
        cus_type.picklist_option AS payment_customer_type_value,
        ppibd.payment_method_external_code AS payment_payment_method_external_code,
        pm_type.picklist_option AS payment_payment_method_value,
        ppibd.bank_country_code AS payment_bank_country_code,
        ppibd.account_owner AS payment_account_owner,
        ppibd.account_number AS payment_account_number,
        ppibd.routing_number AS payment_routing_number,
        ppibd.bank_external_code AS payment_bank_external_code,
        ppibd.iban AS payment_iban,
        ppibd.business_identifier_code AS payment_business_identifier_code,
        ppibd.currency_code AS payment_currency_code,
        ppibd.amount AS payment_amount,
        ppibd.percent AS payment_percent,
        ppibd.pay_sequence AS payment_pay_sequence,
        ppibd.purpose AS payment_purpose,
        ppibd.external_code AS payment_external_code,
        ppibd.attachment_1 AS payment_attachment_1,
        ppibd.completion AS payment_completion, 
        ppibd.is_completed AS payment_is_completed,
        
        psdg.id AS gosi_info_id,
        psdg.social_insurance_grouping AS gosi_social_insurance_grouping,
        psdg.blood_group AS gosi_blood_group,
        psdg.registeration_date AS gosi_registeration_date,
        psdg.end_date AS gosi_end_date,
        psdg.gosi_no AS gosi_gosi_no,
        psdg.non_register_reason AS gosi_non_register_reason,
        psdg.religion AS gosi_religion,
        psdg.completion AS gosi_completion, 
        psdg.is_completed AS gosi_is_completed,

        pd.id AS dependents_info_id,
        pdd.id AS dependent_details_id,
        pd.related_person_id_external AS dependents_related_person_id_external,
        pd.salutation AS dependents_salutation,
        pd.first_name AS dependents_first_name,
        pd.mid_name AS dependents_mid_name,
        pd.last_name AS dependents_last_name,
        pd.display_name AS dependents_display_name,
        pdd.gender AS dependents_gender,
        pd.relation AS dependents_relation,
        pd.dependent_national_id AS dependent_national_id,
        pd.date_of_birth AS dependents_date_of_birth,
        pd.effective_start_date AS dependents_effective_start_date,
        pdd.effective_end_date AS dependents_effective_end_date,
        pdd.category AS dependents_category,
        pdd.cost AS dependents_cost,
        pdd.insurance_company_country AS dependents_insurance_company_country,
        pdd.insurance_policy_number AS dependents_insurance_policy_number,
        pdd.insurance_start_date AS dependents_insurance_start_date,
        pdd.is_beneficiary AS dependents_is_beneficiary,
        pdd.attachment_1 AS dependents_attachment_1,
        pdd.attachment_2 AS dependents_attachment_2,
        pdd.attachment_3 AS dependents_attachment_3,
        pd.completion AS dependents_completion,
        pd.is_completed AS dependents_is_completed,

        emp.id AS employee_id,
        emp.sequence_no,
        emp.employee_id_external, emp.person_id_external,
        emp.recruitment_date, emp.joining_date,
        emp.primary_employee,

        eed.id AS employee_details_id,
        eed.start_date, eed.seq_no, eed.event_reason, ji_event.picklist_option AS event_reason_value,
        eed.position_id, eed.manager_id, eed.custom_string_3,
        manager_person.display_name AS manager_name,
        eed.time_event_type_group_id, eed.contract_end_date,
        eed.contract_type, ji_contract_type.picklist_option AS contract_type_value, 
        eed.default_overtime_compensation_variant,
        eed.employee_class, eed.employment_type,
        eed.end_date, eed.holiday_calendar_code,
        eed.job_code, eed.custom_string2, ji_lg.location_group_name AS location_group_name,
        eed.custom_string11, eed.custom_date4,
        eed.pay_scale_area, eed.pay_scale_group,
        eed.pay_scale_level, eed.pay_scale_type,
        eed.custom_string10, eed.probationary_period,
        eed.probation_period_end_date,
        eed.custom_string4, eed.time_profile_code,
        eed.time_recording_admissibility_code,
        eed.time_recording_profile_code,
        eed.time_recording_variant,
        eed.timezone, eed.workschedule_code,
        eed.working_days_per_week,
        eed.status AS employee_status,

        pos.id AS position_id,
        pos.position_id_external,
        pos.external_name_US AS position_external_name_US,
        pos.external_name_SA AS position_external_name_SA,
        pos.change_reason_id AS position_change_reason_id,
        pos_cr.picklist_option AS position_change_reason_value,
        pos.job_classification_id AS position_job_classification_id,
        jc.job_id_external AS job_id_external,
        jc.name_us AS job_name_us,
        pos.job_level AS position_job_level,
        pos.employee_class_id AS position_employee_class_id,
        pos_ec.picklist_option AS position_employee_class_value,
        pos.grade_id AS position_grade_id,
        pos_grade.grade AS position_grade,
        pos_grade.grade_id_external AS position_grade_id_external,
        pos_grade.category AS position_grade_category,
        pos.target_fte AS position_target_fte,
        pos.is_vacant AS position_is_vacant,
        pos.location_id AS position_location_id,
        pos_location.location_id_external AS position_location_id_external,
        pos_location.location_group_id AS position_location_group_id,
        pos_location_group.location_group_name AS position_location_group_name,
        pos_location.location_name AS position_location_name,
        pos_location.address_line_1 AS position_location_address_line_1,
        pos_location.address_line_2 AS position_location_address_line_2,
        pos_location.address_line_3 AS position_location_address_line_3,
        pos_location.locality AS position_locality,
        pos_location.dependent_locality AS position_dependent_locality,
        pos_location.administrative_area AS position_administrative_area,
        pos_location.sub_administrative_area AS position_sub_administrative_area,
        pos_location.postal_code AS position_postal_code,
        pos_location.country_code AS position_country_code,
        pos_location.sorting_code AS position_sorting_code,
        pos_location.language_code AS position_language_code,
        pos.cost_center_id AS position_cost_center_id,
        pos_cc.cost_center_id_external AS position_cost_center_id_external,
        pos_cc.name_us AS position_cost_center_name_us,
        pos_cc.name_sa AS position_cost_center_name_sa,
        pos_cc.description_us AS position_cost_center_description_us,
        pos_cc.description_sa AS position_cost_center_description_sa,
        pos.multiple_incubment_allowed AS position_multiple_incubment_allowed,
        pos.position_criticality AS position_position_criticality,
        pos.standard_hours AS position_standard_hours,
        pos.cust_country_id AS position_cust_country_id,
        pos.cust_employment_type_id AS position_cust_employment_type_id,
        pos.cust_min AS position_cust_min,
        pos.cust_mid AS position_cust_mid,
        pos.cust_max AS position_cust_max,
        pos.cust_attachment AS position_cust_attachment,
        pos.parent_position_id AS position_parent_position_id,
        pos.effective_start_date AS position_effective_start_date,
        pos.status AS position_status,

        pos.entity_id AS position_entity_id,
        le.entity_id_external AS entity_id_external,
        le.name AS entity_name,
        le.description AS entity_description,
        
        pos.business_unit_id AS position_business_unit_id,
        bu.bu_id_external AS bu_id_external,
        bu.description AS bu_description,

        pos.division_id AS position_division_id,
        division.division_id_external AS division_id_external,
        division.description AS division_description,

        pos.department_id AS position_department_id,
        department.department_id_external AS department_id_external,
        department.description AS department_description,

        pos.section_id AS position_section_id,
        section.section_id_external AS section_id_external,
        section.description AS section_description,

        comp.id AS compensation_id,
        comp.compansation_id_external,
        comp.event_date AS compensation_event_date,
        comp.event_reason AS compensation_event_reason,
        comp_event.picklist_option AS compensation_event_reason_value,
        comp.end_date AS compensation_end_date,
        comp.pay_group_id AS compensation_pay_group_id,
        comp.recurring_pay_component_id,
        comp.non_recurring_pay_component_id,

        rpc.id AS recurring_pay_component_id,
        rpc.recurring_pay_comp_id_external AS recurring_pay_component_id_external,
        rpc.value AS recurring_pay_component_value,
        rpc.currency_code AS recurring_pay_component_currency_code,
        rpc.start_date AS recurring_pay_component_start_date,
        rpc.frequency AS recurring_pay_component_frequency,
        rpc.sequence_no AS recurring_pay_component_sequence_no,
        rpc.enable_end_date AS recurring_pay_component_enable_end_date,
        rpc.end_date AS recurring_pay_component_end_date,
        rpc.note AS recurring_pay_component_note,

        nrpc.id AS non_recurring_pay_component_id,
        nrpc.non_recurring_pay_comp_id_external AS non_recurring_pay_component_id_external,
        nrpc.value AS non_recurring_pay_component_value,
        nrpc.currency_code AS non_recurring_pay_component_currency_code,
        nrpc.pay_date AS non_recurring_pay_component_pay_date,
        nrpc.cost_center_id AS non_recurring_pay_component_cost_center_id,
        nrpc_cc.cost_center_id_external AS non_recurring_pay_component_cost_center_id_external,
        nrpc_cc.name_us AS non_recurring_pay_component_cost_center_name_us,
        nrpc_cc.name_sa AS non_recurring_pay_component_cost_center_name_sa,
        nrpc_cc.description_us AS non_recurring_pay_component_cost_center_description_us,
        nrpc_cc.description_sa AS non_recurring_pay_component_cost_center_description_sa,
        nrpc.no_of_units AS non_recurring_pay_component_no_of_units,
        nrpc.units_of_measure AS non_recurring_pay_component_units_of_measure,
        nrpc.reference_id AS non_recurring_pay_component_reference_id,
        nrpc.start_date AS non_recurring_pay_component_start_date,
        nrpc.end_date AS non_recurring_pay_component_end_date,
        nrpc.note AS non_recurring_pay_component_note,

        ws.schedule_id_external AS work_schedule_id_external,
        ws.external_name AS work_schedule_name,
        ws.starting_date AS work_schedule_starting_date,
        ws.country AS work_schedule_country,
        ws.model As work_schedule_model,
        ws_modal.picklist_option AS work_schedule_model_value,
        ws.cross_midnight_allowed AS work_schedule_cross_midnight_allowed,
        ws.average_hours_per_day AS work_schedule_average_hours_per_day,
        ws.average_hours_per_week AS work_schedule_average_hours_per_week,
        ws.average_hours_per_week AS work_schedule_average_hours_per_week,
        ws.average_hours_per_month AS work_schedule_average_hours_per_month,
        ws.average_hours_per_year AS work_schedule_average_hours_per_year,
        ws.average_working_days_per_week AS work_schedule_average_working_days_per_week,
        ws.shift_classification AS work_schedule_shift_classification,
        ws.time_recording_variant,
        ws_trv.picklist_option AS work_schedule_time_recording_variant_value,
        ws.created_at AS work_schedule_created_at,
        ws.created_by AS work_schedule_created_by,
        ws.updated_at AS work_schedule_updated_at,
        ws.updated_by AS work_schedule_updated_by,

        tp.time_profile_id_external AS time_profile_id_external,
        tp.extrenal_name AS time_profile_name,
        tp.effective_start_date AS time_profile_effective_start_date,
        tp.country AS time_profile_country,
        tp.main_absence_time_type AS time_profile_main_absence_time_type,
        tp.main_attendance_time_type AS time_profile_main_attendance_time_type,
        tp.main_break_time_type AS time_profile_main_break_time_type,
        tp.time_recording_variant AS time_profile_time_recording_variant,
        tp_trv.picklist_option AS time_profile_time_recording_variant_value,
        tp.created_at AS time_profile_created_at,
        tp.created_by AS time_profile_created_by,
        tp.updated_at AS time_profile_updated_at,
        tp.updated_by AS time_profile_updated_by

      FROM ${tables.emp_employee} AS emp
      LEFT JOIN ${tables.emp_employee_details} AS eed
        ON (emp.employee_id_external = eed.employee_id_external AND eed.is_deleted = 2)
      LEFT JOIN ${tables.org_position} AS pos
        ON (eed.position_id = pos.position_id_external AND pos.is_deleted = 2)
      LEFT JOIN ${tables.per_person} AS pp
        ON (emp.person_id_external = pp.person_id_external AND pp.is_deleted = 2)
      LEFT JOIN ${tables.per_personal_info} AS ppi
        ON (pp.id = ppi.person_id AND ppi.is_deleted = 2)
      LEFT JOIN ${tables.per_national_id} AS pni
        ON (pp.id = pni.person_id AND pni.is_deleted = 2)
      LEFT JOIN ${tables.per_driving_license} AS pdl
        ON (pp.id = pdl.person_id AND pdl.is_deleted = 2)
      LEFT JOIN ${tables.per_visa_iqama} AS pvi
        ON (pp.id = pvi.person_id AND pvi.is_deleted = 2)
      LEFT JOIN ${tables.per_passport} AS pppt
        ON (pp.id = pppt.person_id AND pppt.is_deleted = 2)
      LEFT JOIN ${tables.per_email} AS pe
        ON (pp.id = pe.person_id AND pe.is_deleted = 2)
      LEFT JOIN ${tables.per_contact_no} AS pcn
        ON (pp.id = pcn.person_id AND pcn.is_deleted = 2)
      LEFT JOIN ${tables.per_emergency_contact} AS pec
        ON (pp.id = pec.person_id AND pec.is_deleted = 2)
      LEFT JOIN ${tables.per_address} AS pad
        ON (pp.id = pad.person_id AND pad.is_deleted = 2)
      LEFT JOIN ${tables.per_educational_detail} AS ped
        ON (pp.id = ped.person_id AND ped.is_deleted = 2)
      LEFT JOIN ${tables.per_payment_info_bank_details} AS ppibd
        ON (pp.id = ppibd.person_id AND ppibd.is_deleted = 2)
      LEFT JOIN ${tables.per_specific_data_for_gosi} AS psdg
        ON (pp.id = psdg.person_id AND psdg.is_deleted = 2)
      LEFT JOIN ${tables.per_dependent} AS pd
        ON (pp.id = pd.person_id AND pd.is_deleted = 2)
      LEFT JOIN ${tables.per_dependent_details} AS pdd
        ON (pd.id = pdd.related_person_id AND pdd.is_deleted = 2)
      LEFT JOIN ${tables.emp_employee} AS manager
        ON (eed.manager_id = manager.employee_id_external AND manager.is_deleted = 2)
      LEFT JOIN ${tables.per_person} AS manager_person
        ON (manager.person_id_external = manager_person.person_id_external AND manager_person.is_deleted = 2)
      LEFT JOIN ${tables.org_job_classification} AS jc
        ON (pos.job_classification_id = jc.job_id_external AND jc.is_deleted = 2)
      LEFT JOIN ${tables.org_legal_entity} AS le
        ON (pos.entity_id = le.entity_id_external AND le.is_deleted = 2)
      LEFT JOIN ${tables.org_compansation} As comp
        ON (eed.employee_id_external = comp.employee_id_external AND comp.is_deleted = 2)
      LEFT JOIN ${tables.org_recurring_pay_component} AS rpc
        ON (comp.recurring_pay_component_id = rpc.recurring_pay_comp_id_external AND rpc.is_deleted = 2)
      LEFT JOIN ${tables.org_non_recurring_pay_component} AS nrpc
        ON (comp.non_recurring_pay_component_id = nrpc.non_recurring_pay_comp_id_external AND nrpc.is_deleted = 2)
      LEFT JOIN ${tables.org_business_unit} AS bu
        ON (pos.business_unit_id = bu.bu_id_external AND bu.is_deleted = 2)
      LEFT JOIN ${tables.org_division} AS division
        ON (pos.division_id = division.division_id_external AND division.is_deleted = 2)
      LEFT JOIN ${tables.org_department} AS department
        ON (pos.department_id = department.department_id_external AND department.is_deleted = 2)
      LEFT JOIN ${tables.org_section} AS section
        ON (pos.section_id = section.section_id_external AND section.is_deleted = 2)  
      Left JOIN ${tables.org_grade} AS pos_grade
        ON (pos.grade_id = pos_grade.grade_id_external AND pos_grade.is_deleted = 2)
      LEFT JOIN ${tables.org_location_group} AS ji_lg
        ON (eed.custom_string2 = ji_lg.location_group_id_external AND ji_lg.is_deleted = 2)
      Left JOIN ${tables.org_location} AS pos_location
        ON (pos.location_id = pos_location.location_id_external AND pos_location.is_deleted = 2)
      Left JOIN ${tables.org_location_group} AS pos_location_group
        ON (pos_location.location_group_id = pos_location_group.location_group_id_external AND pos_location_group.is_deleted = 2)
      LEFT JOIN ${tables.work_schedule} AS ws
        ON (eed.workschedule_code = ws.schedule_id_external AND ws.is_active = 1)
      LEFT JOIN ${tables.work_schedule_details} AS wsd
        ON (ws.schedule_id_external = wsd.schedule_id_external AND wsd.is_active = 1)
      LEFT JOIN ${tables.day_model} AS dm
        ON (wsd.day_model_id_external = dm.day_model_id_external AND dm.is_active = 1)
      LEFT JOIN ${tables.day_model_segments} AS dms
        ON (dm.day_model_id_external = dms.day_model_id_external AND dms.is_active = 1)
      LEFT JOIN ${tables.time_profile} AS tp
        ON (eed.time_profile_code = tp.time_profile_id_external AND tp.is_active = 1)
      LEFT JOIN ${tables.time_profile_details} AS tpd
        ON (tp.time_profile_id_external = tpd.time_profile_id_external AND tpd.is_active = 1)
      LEFT JOIN ${tables.picklist_master} AS ws_modal
        ON (ws.model = ws_modal.id AND ws_modal.is_deleted = 2)
      LEFT JOIN ${tables.picklist_master} AS ws_trv
        ON (ws.time_recording_variant = ws_trv.id AND ws_trv.is_deleted = 2)
      LEFT JOIN ${tables.picklist_master} AS tp_trv
        ON (tp.time_recording_variant = tp_trv.id AND tp_trv.is_deleted = 2)
      LEFT JOIN ${tables.picklist_master} AS ct
        ON (ped.course_type = ct.id AND ct.is_deleted = 2)
      LEFT JOIN ${tables.picklist_master} AS dt
        ON (ped.degree_title = dt.id AND dt.is_deleted = 2)
      LEFT JOIN ${tables.picklist_master} AS pt
        ON (ppibd.pay_type = pt.id AND pt.is_deleted = 2)
      LEFT JOIN ${tables.picklist_master} AS cus_type
        ON (ppibd.customer_type_external_code = cus_type.id AND cus_type.is_deleted = 2)
      LEFT JOIN ${tables.picklist_master} AS pm_type
        ON (ppibd.payment_method_external_code = pm_type.id AND pm_type.is_deleted = 2)
      LEFT JOIN ${tables.picklist_master} AS er
        ON (eed.event_reason = er.id AND er.is_deleted = 2)
      LEFT JOIN ${tables.picklist_master} AS ji_event
        ON (eed.event_reason = ji_event.id AND ji_event.is_deleted = 2)
      LEFT JOIN ${tables.picklist_master} AS comp_event
        ON (comp.event_reason = comp_event.id AND comp_event.is_deleted = 2)
      LEFT JOIN ${tables.picklist_master} AS ji_contract_type
        ON (eed.contract_type = ji_contract_type.id AND ji_contract_type.is_deleted = 2)
      Left JOIN ${tables.picklist_master} AS pos_cr
        ON (pos.change_reason_id = pos_cr.id AND pos_cr.is_deleted = 2)
      Left JOIN ${tables.picklist_master} AS pos_ec
        ON (pos.employee_class_id = pos_ec.id AND pos_ec.is_deleted = 2)
      LEFT JOIN ${tables.org_cost_center} AS pos_cc
        ON (pos.cost_center_id = pos_cc.cost_center_id_external AND pos_cc.is_deleted = 2)
      LEFT JOIN ${tables.org_cost_center} AS nrpc_cc
        ON (nrpc.cost_center_id = nrpc_cc.cost_center_id_external AND nrpc_cc.is_deleted = 2)
      WHERE emp.is_deleted = 2
      AND emp.employee_id_external = '${employee_id_external}'
    `

    // console.log("personQuery: ", personQuery);

    
    
    const personResult = await performQuery(personQuery);
    const transformedData = await TransformData(personResult);

    console.log("personResult: ", personResult);
    console.log("transformedData: ", transformedData);
    

    return sendResponse(
      res,
      transformedData,
      "Employee Info fetched successfully",
      200
    );
    return sendResponse(
      res,
      personResult,
      "Employee Info fetched successfully",
      200
    );
  } catch (error) {
    console.error("Error while fetching Employee:", error);
    return sendErrorResponse(res, error, "Error while fetching Employee");
  }
};


const TransformData = async (data) => {
  const map = new Map();
  data?.forEach((row) => {
    // Grouping key: adjust if you want a different grouping logic
    const key = `${row.id}-${row.personal_info_id}`;
    if (!map.has(key)) {
      map.set(key, {
        employee_personal_data: {
          personal_info: {
            id: row.id,
            person_id_external: row.person_id_external,
            salutation: row.salutation,
            first_name: row.first_name,
            mid_name: row.mid_name,
            last_name: row.last_name,
            display_name: row.display_name,
            first_name_arabic: row.first_name_arabic,
            mid_name_arabic: row.mid_name_arabic,
            last_name_arabic: row.last_name_arabic,
            display_name_arabic: row.display_name_arabic,
            date_of_birth: row.date_of_birth,
            date_of_birth_hijri: row.date_of_birth_hijri,
            completion: row.completion,
            is_completed: row.is_completed,
            personal_info_id: row.personal_info_id,
            gender: row.gender,
            nationality: row.nationality,
            marital_status: row.marital_status,
            native_prefered_language: row.native_prefered_language,
            attachment_1: row.attachment_1
          },
          national_id_info: [],
          driving_license_info: [],
          visa_iqama_info: [],
          passport_info: [],
          email_info: [],
          contact_info: [],
          emergency_contact_info: [],
          address_info: [],
          educational_info: [],
          payment_info: [],
          gosi_info: [],
          dependents_info: [],
        },
        employment_detail: {
          job_information: [],
          orginization_information: [],
          position_information: [],
          compensation_information: [],
        },
        time_management: {
          work_schedule: [],
          time_profile: [],
        }
      });
    }
    const agg = map.get(key);
    if (row.national_id_info_id !== null) {
      const existingNationalId = agg.employee_personal_data.national_id_info.some(item => item.id === row.national_id_info_id);
      if (!existingNationalId) {
        agg.employee_personal_data.national_id_info.push({
          id: row.national_id_info_id,
          country: row.national_id_country,
          national_id_no: row.national_id_no,
          id_type: row.id_type,
          type: row.id_type == 1 ? "National ID" : row.id_type == 2 ? "Green Card" : "NA",
          issue_date: row.national_id_issue_date,
          issue_date_hijri: row.national_id_issue_date_hijri,
          expiry_date: row.national_id_expiry_date,
          expiry_date_hijri: row.national_id_expiry_date_hijri,
          attachment_1: row.national_id_attachment_1,
          attachment_2: row.national_id_attachment_2,
          completion: row.national_id_completion,
          is_completed: row.national_id_is_completed
        });
      }    
    }
    if (row.driving_license_info_id !== null) {
      const existingDrivingLicense = agg.employee_personal_data.driving_license_info.some(item => item.id === row.driving_license_info_id);
      if (!existingDrivingLicense) {
        agg.employee_personal_data.driving_license_info.push({
          id: row.driving_license_info_id,
          country: row.driving_license_country,
          license_no: row.license_no,
          issue_date: row.driving_license_issue_date,
          issue_date_hijri: row.driving_license_issue_date_hijri,
          expiry_date: row.driving_license_expiry_date,
          expiry_date_hijri: row.driving_license_expiry_date_hijri,
          attachment_1: row.driving_license_attachment_1,
          attachment_2: row.driving_license_attachment_2,
          completion: row.driving_license_completion,
          is_completed: row.driving_license_is_completed
        });
      }
    }
    if (row.visa_iqama_info_id !== null) {
      const existingVisaIqama = agg.employee_personal_data.visa_iqama_info.some(item => item.id === row.visa_iqama_info_id);
      if (!existingVisaIqama) {
        agg.employee_personal_data.visa_iqama_info.push({
          id: row.visa_iqama_info_id,
          country: row.visa_iqama_country,
          document_no: row.document_no,
          document_type: row.visa_iqama_document_type,
          type: row.visa_iqama_document_type == 1 ? "Visa" : row.visa_iqama_document_type == 2 ? "Iqama" : "NA",
          boarder_no: row.boarder_no,
          issue_date: row.visa_iqama_issue_date,
          issue_date_hijri: row.visa_iqama_issue_date_hijri,
          issue_place: row.visa_iqama_issue_place,
          issuing_authority: row.visa_iqama_issuing_authority,
          validated: row.visa_iqama_validated,
          expiry_date: row.visa_iqama_expiry_date,
          expiry_date_hijri: row.visa_iqama_expiry_date_hijri,
          attachment_1: row.visa_iqama_attachment_1,
          attachment_2: row.visa_iqama_attachment_2,
          notes: row.visa_iqama_notes,
          completion: row.visa_iqama_completion,
          is_completed: row.visa_iqama_is_completed
        });
      }
    }
    if (row.passport_info_id !== null) {
      const existingPassport = agg.employee_personal_data.passport_info.some(item => item.id === row.passport_info_id);
      if (!existingPassport) {
        agg.employee_personal_data.passport_info.push({
          id: row.passport_info_id,
          country: row.passport_country,
          passport_no: row.passport_no,
          issue_date: row.passport_issue_date,
          issue_date_hijri: row.passport_issue_date_hijri,
          issue_place: row.passport_issue_place,
          issuing_authority: row.passport_issuing_authority,
          validated: row.passport_validated,
          expiry_date: row.passport_expiry_date,
          expiry_date_hijri: row.passport_expiry_date_hijri,
          attachment_1: row.passport_attachment_1,
          attachment_2: row.passport_attachment_2,
          completion: row.passport_completion,
          is_completed: row.passport_is_completed,
          notes: row.passport_notes
        });
      }
    }
    if (row.email_info_id !== null) {
      const existingEmail = agg.employee_personal_data.email_info.some(item => item.id === row.email_info_id);
      if (!existingEmail) {
        agg.employee_personal_data.email_info.push({
          id: row.email_info_id,
          email_type: row.email_type,
          type: row.email_type == 1 ? "Personal" : row.email_type == 2 ? "Work" : "NA",
          email: row.email,
          is_primary: row.email_is_primary,
          is_verified: row.email_is_verified
        });
      }
    }
    if (row.contact_info_id !== null) {
      const existingContact = agg.employee_personal_data.contact_info.some(item => item.id === row.contact_info_id);
      if (!existingContact) {
        agg.employee_personal_data.contact_info.push({
          id: row.contact_info_id,
          country_code: row.country_code,
          contact_no: row.contact_no,
          contact_type: row.contact_type,
          type: row.contact_type == 1 ? "Mobile" : row.contact_type == 2 ? "Home" : row.contact_type == 3 ? "Work" : "NA",
          is_primary: row.contact_is_primary
        });
      }
    }
    if (row.emergency_contact_info_id !== null) {
      const existingEmergencyContact = agg.employee_personal_data.emergency_contact_info.some(item => item.id === row.emergency_contact_info_id);
      if (!existingEmergencyContact) {
        agg.employee_personal_data.emergency_contact_info.push({
          id: row.emergency_contact_info_id,
          name: row.emergency_contact_name,
          relation: row.emergency_contact_relation,
          phone: row.emergency_contact_phone,
          alternate_phone: row.emergency_contact_alternate_phone,
          address_line_1: row.emergency_contact_address_line_1,
          address_line_2: row.emergency_contact_address_line_2,
          address_line_3: row.emergency_contact_address_line_3,
          locality: row.emergency_contact_locality,
          dependent_locality: row.emergency_contact_dependent_locality,
          administrative_area: row.emergency_contact_administrative_area,
          sub_administrative_area: row.emergency_contact_sub_administrative_area,
          postal_code: row.emergency_contact_postal_code,
          country_code: row.emergency_contact_country_code,
          notes: row.emergency_contact_notes,
          same_address: row.emergency_contact_same_address,
          completion: row.emergency_contact_completion,
          is_completed: row.emergency_contact_is_completed
        });
      }
    }
    if (row.address_info_id !== null) {
      const existingAddress = agg.employee_personal_data.address_info.some(item => item.id === row.address_info_id);
      if (!existingAddress) {
        agg.employee_personal_data.address_info.push({
          id: row.address_info_id,
          address_type: row.address_type,
          type: row.address_type == 1 ? "Permenent" : row.address_type == 2 ? "Current" : "NA",
          address_line_1: row.address_line_1,
          address_line_2: row.address_line_2,
          address_line_3: row.address_line_3,
          locality: row.locality,
          dependent_locality: row.dependent_locality,
          administrative_area: row.administrative_area,
          sub_administrative_area: row.sub_administrative_area,
          postal_code: row.postal_code,
          sorting_code: row.sorting_code,
          language_code: row.language_code,
          is_primary: row.address_is_primary,
          completion: row.address_completion,
          is_completed: row.address_is_completed
        });
      }
    }
    if (row.educational_info_id !== null) {
      const existingEducational = agg.employee_personal_data.educational_info.some(item => item.id === row.educational_info_id);
      if (!existingEducational) {
        agg.employee_personal_data.educational_info.push({
          id: row.educational_info_id,
          course_type: row.educational_course_type,
          course_type_value: row.educational_course_type_value,
          document_name: row.educational_document_name,
          degree_title: row.educational_degree_title,
          degree_title_value: row.educational_degree_title_value,
          grade: row.educational_grade,
          institute_name: row.educational_institute_name,
          major_subject: row.educational_major_subject,
          document_no: row.educational_document_no,
          start_date: row.educational_start_date,
          end_date: row.educational_end_date,
          status: row.educational_status,
          attachment_1: row.educational_attachment_1,
          attachment_2: row.educational_attachment_2,
          completion: row.educational_completion,
          is_completed: row.educational_is_completed
        });
      }
    }
    if (row.payment_info_id !== null) {
      const existingPayment = agg.employee_personal_data.payment_info.some(item => item.id === row.payment_info_id);
      if (!existingPayment) {
        agg.employee_personal_data.payment_info.push({
          id: row.payment_info_id,
          effective_start_date: row.payment_effective_start_date,
          pay_type: row.payment_pay_type,
          pay_type_value: row.payment_pay_type_value,
          customer_type_external_code: row.payment_customer_type_external_code,
          customer_type_value: row.payment_customer_type_value,
          payment_method_external_code: row.payment_payment_method_external_code,
          payment_method_value: row.payment_payment_method_value,
          bank_country_code: row.payment_bank_country_code,
          account_owner: row.payment_account_owner,
          account_number: row.payment_account_number,
          routing_number: row.payment_routing_number,
          bank_external_code: row.payment_bank_external_code,
          iban: row.payment_iban,
          business_identifier_code: row.payment_business_identifier_code,
          currency_code: row.payment_currency_code,
          amount: row.payment_amount,
          percent: row.payment_percent,
          pay_sequence: row.payment_pay_sequence,
          purpose: row.payment_purpose,
          external_code: row.payment_external_code,
          attachment_1: row.payment_attachment_1,
          completion: row.payment_completion,
          is_completed: row.payment_is_completed
        });
      }
    }
    if (row.gosi_info_id !== null) {
      const existingGosi = agg.employee_personal_data.gosi_info.some(item => item.id === row.gosi_info_id);
      if (!existingGosi) {
        agg.employee_personal_data.gosi_info.push({
          id: row.gosi_info_id,
          social_insurance_grouping: row.gosi_social_insurance_grouping,
          blood_group: row.gosi_blood_group,
          registeration_date: row.gosi_registeration_date,
          end_date: row.gosi_end_date,
          gosi_no: row.gosi_gosi_no,
          non_register_reason: row.gosi_non_register_reason,
          religion: row.gosi_religion,
          completion: row.gosi_completion,
          is_completed: row.gosi_is_completed
        });
      }
    }
    if (row.dependents_info_id !== null) {
      const existingDependents = agg.employee_personal_data.dependents_info.some(item => item.id === row.dependents_info_id);
      if (!existingDependents) {
        agg.employee_personal_data.dependents_info.push({
          id: row.dependents_info_id,
          dependent_details_id: row.dependent_details_id,
          related_person_id_external: row.dependents_related_person_id_external,
          salutation: row.dependents_salutation,
          first_name: row.dependents_first_name,
          mid_name: row.dependents_mid_name,
          last_name: row.dependents_last_name,
          display_name: row.dependents_display_name,
          gender: row.dependents_gender,
          relation: row.dependents_relation,
          dependent_national_id: row.dependent_national_id,
          date_of_birth: row.dependents_date_of_birth,
          effective_start_date: row.dependents_effective_start_date,
          effective_end_date: row.dependents_effective_end_date,
          category: row.dependents_category,
          cost: row.dependents_cost,
          insurance_company_country: row.dependents_insurance_company_country,
          insurance_policy_number: row.dependents_insurance_policy_number,
          insurance_start_date: row.dependents_insurance_start_date,
          is_beneficiary: row.dependents_is_beneficiary,
          attachment_1: row.dependents_attachment_1,
          attachment_2: row.dependents_attachment_2,
          attachment_3: row.dependents_attachment_3,
          completion: row.dependents_completion,
          is_completed: row.dependents_is_completed
        });
      }
    }
    // Job Information
    if (row.employee_id !== null) {
      console.log("agg.employment_detail.job_information: ", agg.employment_detail.job_information);
      
      const existingJobInfo = agg.employment_detail.job_information.some(item => item.id === row.employee_id);
      if (!existingJobInfo) {
        agg.employment_detail.job_information.push({
          id: row.employee_id,
          job_name: row.job_name_us + ` (${row.job_id_external})`,
          recruitment_date: row.recruitment_date == "null" ? '---' : row.recruitment_date,
          joining_date: row.joining_date == "null" ? '---' : row.joining_date,
          primary_employee: row.primary_employee == 1 ? "Yes" : "No",
          start_date: row.start_date == "null" ? '---' : row.start_date,
          event_reason: row.event_reason_value,
          position_name: row.position_external_name_US + ` (${row.position_id_external})`,
          manager_name: row.manager_name == "null" ? '---' : row.manager_name + ` (${row.manager_id})`,
          custom_string_3: row.custom_string_3,
          contract_end_date: row.contract_end_date,
          contract_type: row.contract_type,
          contract_type_value: row.contract_type_value,
          employee_class: row.employee_class,
          employment_type: row.employment_type,
          end_date: row.end_date,
          status: row.employee_status == 1 ? "Active" : "Inactive",
        })
      }
    }
    // Position Information
    if (row.position_id !== null) {
      const existingPositionInfo = agg.employment_detail.position_information.some(item => item.id === row.position_id);
      if (!existingPositionInfo) {
        agg.employment_detail.position_information.push({
          id: row.position_id,
          external_name_US: row.position_external_name_US + ` (${row.position_id_external})`,
          external_name_SA: ` ${row.position_external_name_SA == "null"? "---" : row.position_external_name_SA} `,
          change_reason: row.position_change_reason_value,
          job_name_us: row.job_name_us + ` (${row.job_id_external})`,
          job_level: row.position_job_level,
          employee_class: row.position_employee_class_value,
          grade: row.position_grade + ` (${row.position_grade_id_external})`,
          grade_category: row.position_grade_category == 1 ? "Managerial" : row.position_grade_category == 2 ? "Non Managerial" : "NA",
          target_fte: row.position_target_fte,
          is_vacant_value: row.position_is_vacant == 1 ? "Yes" : "No",
          location_group: row.position_location_group_name + ` (${row.position_location_group_id})`,
          location_name: row.position_location_name + ` (${row.position_location_id_external})`,
          location_address_line_1: row.position_location_address_line_1,
          location_address_line_2: row.position_location_address_line_2,
          location_address_line_3: row.position_location_address_line_3,
          location_locality: row.position_locality,
          location_dependent_locality: row.position_dependent_locality,
          location_administrative_area: row.position_administrative_area,
          location_sub_administrative_area: row.position_sub_administrative_area,
          location_postal_code: row.position_postal_code,
          location_country_code: row.position_country_code,
          location_sorting_code: row.position_sorting_code,
          location_language_code: row.position_language_code,
          cost_center_name_us: row.position_cost_center_name_us + ` (${row.position_cost_center_id_external})`,
          cost_center_name_sa: row.position_cost_center_name_sa,
          cost_center_description_us: row.position_cost_center_description_us,
          cost_center_description_sa: row.position_cost_center_description_sa,
          multiple_incubment_allowed: row.position_multiple_incubment_allowed == 1 ? "Yes" : "No",
          position_criticality: row.position_position_criticality == 1 ? "High" : "Low",
          standard_hours: row.position_standard_hours,
          cust_country: row.position_cust_country_id,
          cust_employment_type_id: row.position_cust_employment_type_id == "null" ? '---' : row.position_cust_employment_type_id,
          cust_min: row.position_cust_min == "null" ? '---' : row.position_cust_min,
          cust_mid: row.position_cust_mid == "null" ? '---' : row.position_cust_mid,
          cust_max: row.position_cust_max == "null" ? '---' : row.position_cust_max,
          cust_attachment: row.position_cust_attachment == "null" ? '---' : row.position_cust_attachment,
          effective_start_date: row.position_effective_start_date == "null" ? '---' : row.position_effective_start_date,
          status: row.position_status == 1 ? "Active" : "Inactive"
        });
      }
    }
    // Organization Information
    if (row.position_id !== null) {
      const existingOrginizationInfo = agg.employment_detail.orginization_information.some(item => item.id === row.position_id);
      if (!existingOrginizationInfo) {
        agg.employment_detail.orginization_information.push({
          id: row.position_id,
          legal_entity: row.entity_name + ' (' + row.entity_id_external + ')',
          entity_description: row.entity_description,
          business_unit: row.bu_description + ' (' + row.bu_id_external + ')',
          division: row.division_description + ' (' + row.division_id_external + ')',
          department: row.department_description + ' (' + row.department_id_external + ')',
          section: row.section_description + ' (' + row.section_id_external + ')'
        });
      }
    }
    // Compensation Information
    if (row.compensation_id !== null) {
      const existingCompensationInfo = agg.employment_detail.compensation_information.some(item => item.id === row.compensation_id);
      if (!existingCompensationInfo) {
        agg.employment_detail.compensation_information.push({
          id: row.compensation_id,
          compansation_id_external: row.compansation_id_external,
          event_date: row.compensation_event_date,
          event_reason: row.compensation_event_reason,
          event_reason_value: row.compensation_event_reason_value,
          end_date: row.compensation_end_date,
          pay_group_id: row.compensation_pay_group_id,
          recurring_pay_component: {
            recurring_pay_component_id: row.recurring_pay_component_id,
            recurring_pay_component_id_external: row.recurring_pay_component_id_external,
            value: row.recurring_pay_component_value,
            currency_code: row.recurring_pay_component_currency_code,
            start_date: row.recurring_pay_component_start_date,
            frequency: row.recurring_pay_component_frequency,
            sequence_no: row.recurring_pay_component_sequence_no,
            enable_end_date: row.recurring_pay_component_enable_end_date,
            end_date: row.recurring_pay_component_end_date,
            note: row.recurring_pay_component_note
          },
          non_recurring_pay_component: {
            non_recurring_pay_component_id: row.non_recurring_pay_component_id,
            non_recurring_pay_component_id_external: row.non_recurring_pay_component_id_external,
            value: row.non_recurring_pay_component_value,
            currency_code: row.non_recurring_pay_component_currency_code,
            pay_date: row.non_recurring_pay_component_pay_date,
            cost_center_id: row.non_recurring_pay_component_cost_center_id,
            cost_center_id_external: row.non_recurring_pay_component_cost_center_id_external,
            cost_center_name_us: row.non_recurring_pay_component_cost_center_name_us,
            cost_center_name_sa: row.non_recurring_pay_component_cost_center_name_sa,
            cost_center_description_us: row.non_recurring_pay_component_cost_center_description_us,
            cost_center_description_sa: row.non_recurring_pay_component_cost_center_description_sa,
            no_of_units: row.non_recurring_pay_component_no_of_units,
            units_of_measure: row.non_recurring_pay_component_units_of_measure,
            reference_id: row.non_recurring_pay_component_reference_id,
            start_date: row.non_recurring_pay_component_start_date,
            end_date: row.non_recurring_pay_component_end_date,
            note: row.non_recurring_pay_component_note
          }
        });
      }
    }
    // work schedule
    if (row.work_schedule_id_external !== null) {
      const existingWorkSchedule = agg.time_management.work_schedule.some(item => item.id === row.work_schedule_id_external);
      if (!existingWorkSchedule) {
        agg.time_management.work_schedule.push({
          id: row.work_schedule_id_external,
          external_name: row.work_schedule_name,
          starting_date: row.work_schedule_starting_date,
          country: row.work_schedule_country,
          model: row.work_schedule_model_value,
          cross_midnight_allowed: row.work_schedule_cross_midnight_allowed == 1 ? "Yes" : "No",
          average_hours_per_day: row.work_schedule_average_hours_per_day,
          average_hours_per_week: row.work_schedule_average_hours_per_week,
          average_hours_per_month: row.work_schedule_average_hours_per_month,
          average_hours_per_year: row.work_schedule_average_hours_per_year,
          average_working_days_per_week: row.work_schedule_average_working_days_per_week,
          shift_classification: row.work_schedule_shift_classification,
          time_recording_variant: row.work_schedule_time_recording_variant_value,
          created_by: row.work_schedule_created_by,
          created_at: row.work_schedule_created_at,
          updated_by: row.work_schedule_updated_by,
          updated_at: row.work_schedule_updated_at,
          schedule_details: []          
        });
        // const existingScheduleDetail = agg.time_management.work_schedule.find(item => item.id === row.work_schedule_id_external)?.schedule_details
          // .some(detail => detail.id === row.day_model_segment_id_external);
      }
    }
    if (row.time_profile_id_external !== null) {
      const existingTimeProfile = agg.time_management.time_profile.some(item => item.id === row.time_profile_id_external);
      if (!existingTimeProfile) {
        agg.time_management.time_profile.push({
          id: row.time_profile_id_external,
          external_name: row.time_profile_name,
          effective_start_date: row.time_profile_effective_start_date,
          country: row.time_profile_country,
          main_absence_time_type: row.time_profile_main_absence_time_type,
          main_absence_time_type: row.time_profile_main_absence_time_type,
          main_break_time_type: row.time_profile_main_break_time_type,
          time_recording_variant: row.time_profile_time_recording_variant_value,
          created_by: row.work_schedule_created_by,
          created_at: row.work_schedule_created_at,
          updated_by: row.work_schedule_updated_by,
          updated_at: row.work_schedule_updated_at,
          available_time_type: []          
        });
        // const existingScheduleDetail = agg.time_management.work_schedule.find(item => item.id === row.work_schedule_id_external)?.schedule_details
          // .some(detail => detail.id === row.day_model_segment_id_external);
      }
    }
  });
  return Array.from(map.values());
}