const {
  sendErrorResponse,
  sendResponse,
} = require("../../utils/index.js");
const { tables } = require("../../utils/tables.js");
const { performQuery } = require("../../utils/db.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../../functions/getTimezone.js");


// CREATE Unit
module.exports.createUnit = async (req, res) => {
  try {
    const {
      unit_id_external,
      section_id,
      effective_start_date,
      description,
      description_arabic,
    } = req.body;

    if (!unit_id_external || !section_id || !effective_start_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check section_id exists in section table
    const isSectionID = await performQuery(
      `SELECT * FROM ${tables.org_section} WHERE id = ? AND is_deleted = 2`,
      [section_id]
    );

    if (isSectionID.length === 0) {
      const errorMessage = "Section ID not found against Section table";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check for duplicate unit_id_external
    const existingUnit = await performQuery(
      `SELECT id FROM ${tables.org_unit} WHERE unit_id_external = ? AND is_deleted = 2`,
      [unit_id_external]
    );
    if (existingUnit && existingUnit.length > 0) {
      const errorMessage = "Unit ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get system time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert Record
    await performQuery(`INSERT INTO ${tables.org_unit} SET ?`, {
      sequence_no : 1,
      unit_id_external,
      section_id,
      effective_start_date,
      description,
      description_arabic,
      created_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    });

    return sendResponse(res, {}, "Unit External created successfully", 201);

  } catch (error) {
    console.error("Error while creating Unit External:", error);
    return sendErrorResponse(res, error, "Error while creating Unit External");
  }
};


// READ Unit(s)
module.exports.getUnit = async (req, res) => {
  try {
    const { 
      id, unit_id_external, section_id, 
      section_id_external, department_id, department_id_external, 
      division_id, division_id_external, business_unit_id, 
      bu_id_external, entity_id, entity_id_external 
    } = req.query;

    let query = `
      SELECT 
        unit.*,
        section.section_id_external, section.description AS section_description,
        department.department_id_external, department.description AS department_description,
        division.division_id_external, division.description AS division_description,
        bu.bu_id_external, bu.description AS business_unit_description,
        le.entity_id_external, le.name AS legal_entity_name
      FROM ${tables.org_unit} unit
      LEFT JOIN ${tables.org_section} section ON unit.section_id = section.id AND section.is_deleted = 2
      LEFT JOIN ${tables.org_department} department ON section.department_id = department.id AND department.is_deleted = 2
      LEFT JOIN ${tables.org_division} division ON department.division_id = division.id AND division.is_deleted = 2
      LEFT JOIN ${tables.org_business_unit} bu ON division.business_unit_id = bu.id AND bu.is_deleted = 2
      LEFT JOIN ${tables.org_legal_entity} le ON bu.entity_id = le.id AND le.is_deleted = 2
      WHERE unit.is_deleted = 2
    `;

    if (id) query += ` AND unit.id = ${id}`;
    if (unit_id_external) query += ` AND unit.unit_id_external = '${unit_id_external}'`;
    if (section_id) query += ` AND unit.section_id = ${section_id}`;
    if (section_id_external) query += ` AND section.section_id_external = '${section_id_external}'`;
    if (department_id) query += ` AND department.id = ${department_id}`;
    if (department_id_external) query += ` AND department.department_id_external = '${department_id_external}'`;
    if (division_id) query += ` AND division.id = ${division_id}`;
    if (division_id_external) query += ` AND division.division_id_external = '${division_id_external}'`;
    if (business_unit_id) query += ` AND bu.id = ${business_unit_id}`;
    if (bu_id_external) query += ` AND bu.bu_id_external = '${bu_id_external}'`;
    if (entity_id) query += ` AND le.id = ${entity_id}`;
    if (entity_id_external) query += ` AND le.entity_id_external = '${entity_id_external}'`;

    query += ` ORDER BY unit.id DESC`;

    const units = await performQuery(query);
    return sendResponse(res, units, "Unit(s) fetched successfully", 200);

  } catch (error) {
    console.error("Error while fetching Unit(s):", error);
    return sendErrorResponse(res, error, "Error while fetching Unit(s)");
  }
};


// UPDATE Unit
module.exports.updateUnit = async (req, res) => {
  try {
    const {
      id,
      unit_id_external,
      section_id,
      effective_start_date,
      description,
      description_arabic,
      status,
    } = req.body;

    if (!id || !unit_id_external || !section_id || !effective_start_date) {
      const errorMessage = "Missing required fields";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if record exists
    const existingRecord = await performQuery(
      `SELECT * FROM ${tables.org_unit} WHERE id = ? AND is_deleted = 2`,
      [id]
    );
    if (!existingRecord || existingRecord.length === 0) {
      const errorMessage = "Unit record not found";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }
    const oldRecord = existingRecord[0];

    // Check for duplicate unit_id_external
    const duplicateCheck = await performQuery(
      `SELECT id FROM ${tables.org_unit} WHERE unit_id_external = ? AND id <> ? AND is_deleted = 2`,
      [unit_id_external, id]
    );
    if (duplicateCheck && duplicateCheck.length > 0) {
      const errorMessage = "Unit ID already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Deactivate existing record to maintain history
        await performQuery(
            `UPDATE ${tables.org_unit} SET ? WHERE id = ?`,
            [{
                is_deleted: 1,
                updated_by: req?.user?.id,
                updated_at: currentTime
            },id]
        );

    // Insert Update record
    await performQuery(`INSERT INTO ${tables.org_unit} SET ?`, [{
      sequence_no: Number(oldRecord.sequence_no) + 1,
      effective_start_date: effective_start_date || oldRecord.effective_start_date,
      description: description || oldRecord.description,
      description_arabic: description_arabic || oldRecord.description_arabic,
      status: status || oldRecord.status,
      updated_at: currentTime,
      created_by: req?.user?.id,
      updated_at: currentTime,
      updated_by: req?.user?.id,
    }, id]);

    return sendResponse(res, {}, "Unit External updated successfully", 200);
  } catch (error) {
    console.error("Error while updating Unit External:", error);
    return sendErrorResponse(res, error, "Error while updating Unit External");
  }
};


// DELETE Unit (Soft Delete)
// module.exports.deleteUnit = async (req, res) => {
//   try {
//     const { id } = req.query;

//     if (!id) {
//       const errorMessage = "Unit ID is required";
//       return sendErrorResponse(res, errorMessage, errorMessage);
//     }

//     // Get System Time
//     const systemTime = await getSystemTime();
//     const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

//     // Soft delete
//     await performQuery(
//       `UPDATE ${tables.org_unit} SET ? WHERE id = ?`,
//       [
//         {
//           is_deleted: 1,
//           updated_at: currentTime,
//           updated_by: req?.user?.id,
//         },
//         id,
//       ]
//     );

//     return sendResponse(res, {}, "Unit External deleted successfully", 200);
//   } catch (error) {
//     console.error("Error while deleting Unit External:", error);
//     return sendErrorResponse(res, error, "Error while deleting Unit External");
//   }
// };
