const { time } = require("node:console");
const { getSystemTime } = require("../functions/getTimezone");
const { logger } = require("../helper/logger.helper");
const { performQuery } = require("../utils/db");
const { tables } = require("../utils/tables");
const moment = require("moment-timezone");

module.exports.timeSheetManagementCronJob = async () => {
    try {
        logger.info("Time Sheet Management Cron Job started.");

        // Get current system time
        const systemtime = await getSystemTime();
        const currenttime = moment(systemtime).subtract(0, 'days').format("YYYY-MM-DD HH:mm:ss");
        const dayOfWeek = moment(currenttime).weekday(); // 0 (Sunday) to 6 (Saturday)
        // Define date range for the time sheet
        const dateFrom = moment(currenttime).subtract(dayOfWeek, 'days').format("YYYY-MM-DD");
        const dateTo = moment(dateFrom).add(6, 'days').format("YYYY-MM-DD");
        
        // Only Create Time Sheets on Sundays
        if(moment(dateFrom).weekday() === 0){    
            const timeSheet = [];

            

            // Fetch Data that need to be processed
            const requiredDataQuery = `
                Select 
                    ed.employee_id_external,
                    ws.schedule_id_external,
                    tp.time_profile_id_external,
                    wsd.day, dm.planned_hours, 
                    mit.start_time AS min_in_time, mot.end_time AS max_out_time,
                    ts.time_sheet_id_external
                From ${tables.emp_employee_details} ed
                LEFT JOIN ${tables.work_schedule} ws ON ws.schedule_id_external = ed.workschedule_code AND ws.is_active = 1
                LEFT JOIN ${tables.time_profile} tp ON tp.time_profile_id_external = ed.time_profile_code AND tp.is_active = 1
                LEFT JOIN ${tables.work_schedule_details} wsd ON wsd.schedule_id_external = ws.schedule_id_external AND wsd.is_active = 1
                LEFT JOIN ${tables.time_sheet} ts ON ts.employee_id_external = ed.employee_id_external AND ts.date_from = '${dateFrom}' AND ts.date_to = '${dateTo}' AND ts.is_active = 1
                LEFT JOIN ${tables.day_model} dm ON dm.day_model_id_external = wsd.day_model_id_external AND dm.is_active = 1
                LEFT JOIN (
                    SELECT day_model_id_external, MIN(start_time) AS start_time
                    FROM ${tables.day_model_segments}
                    WHERE is_active = 1
                    GROUP BY day_model_id_external
                ) mit ON mit.day_model_id_external = dm.day_model_id_external
                LEFT JOIN (
                    SELECT day_model_id_external, MAX(end_time) AS end_time
                    FROM ${tables.day_model_segments}
                    WHERE is_active = 1
                    GROUP BY day_model_id_external
                ) mot ON mot.day_model_id_external = dm.day_model_id_external
                Where 
                    ed.is_deleted = 2 AND 
                    ed.workschedule_code IS NOT NULL AND 
                    ed.time_profile_code IS NOT NULL
                ORDER BY 
                    ed.employee_id_external ASC,
                    wsd.day ASC
            `;
            logger.info("Executing Required Data Query for Time Sheet Management Cron Job: ", requiredDataQuery);
            

            // Fetch required data
            let requiredData = await performQuery(requiredDataQuery);
            
            // Filter out records which already have time_sheet_id_external (i.e., Time Sheet already created)
            requiredData = requiredData.filter(rd => !rd.time_sheet_id_external);
            logger.info(`Fetched ${requiredData.length} records for Time Sheet creation. Data: `, requiredData);

            // Process each record and prepare Time Sheets
            await requiredData.map(async rd => {
            const time_sheet_id_external = `TS_${rd.employee_id_external}_${moment(dateFrom).format("MMM-DD-YYYY")}_${moment(dateTo).format("MMM-DD-YYYY")}`;
                
                // If Time Sheet for this ID doesn't exist, create it
                if(!timeSheet[time_sheet_id_external]){
                    timeSheet[time_sheet_id_external] = {
                        time_sheet_id_external: time_sheet_id_external,
                        external_name: `Time Sheet for ${moment(dateFrom).format("MMM-DD-YYYY")} to ${moment(dateTo).format("MMM-DD-YYYY")}`,
                        date_from: dateFrom,
                        date_to: dateTo,
                        employee_id_external: rd.employee_id_external,
                        time_profile_id_external: rd.time_profile_id_external,
                        schedule_id_external: rd.schedule_id_external,
                        planned_working_time: 0,    
                        recorded_working_time: 0,
                        time_sheet_details: []
                    };
                }
                // Add planned hours to the Time Sheet and prepare details
                timeSheet[time_sheet_id_external].planned_working_time += (rd.planned_hours) * 60; // Convert hours to minutes
                await timeSheet[time_sheet_id_external].time_sheet_details.push({
                    time_sheet_id_external: time_sheet_id_external,
                    day: rd.day,
                    date: moment(dateFrom).add(rd.day, 'days').format("YYYY-MM-DD"),
                    planned_time: (rd.planned_hours) * 60 || 0,
                    recorded_time: 0,
                    recordings: 0
                });
            });

            // Insert Time Sheets and their Details in bulk
            let insertTimeSheetQuery = `
                INSERT INTO ${tables.time_sheet} 
                (
                    time_sheet_id_external, external_name, date_from, date_to, employee_id_external, 
                    time_profile_id_external, schedule_id_external, planned_working_time, 
                    recorded_working_time, created_by, created_at, updated_by, updated_at
                ) VALUES 
            `
            // Insert Time Sheet Details in bulk
            let insertTimeSheetDetailsQuery = `
                INSERT INTO ${tables.time_sheet_detail} 
                (
                    time_sheet_id_external, day, date, planned_time, recorded_time, recordings, 
                    created_by, created_at, updated_by, updated_at
                ) VALUES 
            `
            // Prepare bulk insert queries
            Object.values(timeSheet).forEach(ts => {

                // Prepare Time Sheet insert query
                insertTimeSheetQuery += `(
                    '${ts.time_sheet_id_external}', '${ts.external_name}', '${ts.date_from}', '${ts.date_to}', '${ts.employee_id_external}', 
                    '${ts.time_profile_id_external}', '${ts.schedule_id_external}', ${ts.planned_working_time}, 
                    ${ts.recorded_working_time}, 0, '${currenttime}', 0, '${currenttime}'
                ),`;

                // Prepare Time Sheet Details insert query
                ts.time_sheet_details.forEach(tsd => {
                    insertTimeSheetDetailsQuery += `(
                        '${tsd.time_sheet_id_external}', '${tsd.day}', '${tsd.date}', ${tsd.planned_time}, '${tsd.recorded_time}', '${tsd.recordings}', 
                        0, '${currenttime}', 0, '${currenttime}'
                    ),`;
                });
            });

            // Finalize queries by removing trailing commas and adding semicolons
            insertTimeSheetQuery = insertTimeSheetQuery.slice(0, -1) + ";";
            insertTimeSheetDetailsQuery = insertTimeSheetDetailsQuery.slice(0, -1) + ";";

            // Execute bulk insert queries if there are records to insert
            if(insertTimeSheetQuery.length > 420){
                logger.info("Inserting Time Sheets in Bulk: ", insertTimeSheetQuery);
                await performQuery(insertTimeSheetQuery);
                logger.info("Time Sheets inserted successfully.");
            }
            if(insertTimeSheetDetailsQuery.length > 300){
                logger.info("Inserting Time Sheet Details in Bulk: ", insertTimeSheetDetailsQuery);
                await performQuery(insertTimeSheetDetailsQuery);
                logger.info("Time Sheet Details inserted successfully.");
            }
        }


        
    } catch (error) {
        console.log("Error in Time Sheet Management Cron Job: ", error);
        logger.error("Error in Time Sheet Management Cron Job: ", error);
    }
}